#!/usr/bin/perl -w
package s_records;    # Our package

use strict;   # The party's over

# ###################################################################
# ###################################################################
# ###################################################################
# ########
# ########  S Record module - Utilities for reading and writing S records
# ########      This module is based on code written by David Siu
# ########      Adaptations were made by Jeremy Wade


# *********** Package module interface stuff
use Exporter;
our @ISA = ('Exporter');
our $VERSION=1.0;

our @EXPORT=qw(
  &decode_s_records
  &process_line
  &generate_s_line
  &generate_s_block

  $addr_base
  $target_offset
  $line_count
  $s_record_strict
  $this_s_record
  $bytes_read

  S_REC_SUCCESS
  S_REC_NOHEADER
  S_REC_NOFOOTER
  S_REC_INVALID_TYPE
  S_REC_MALFORMED
  S_REC_BAD_CS
  S_REC_TOO_LARGE
  S_REC_UNSORTED
  MAX_BINARY_SIZE
  S_REC_RTN
  S_REC_NO_FOOTER
  S_REC_NO_HEADER
  S_REC_UNKNOWN
  );

use constant S_REC_SUCCESS      => 0;
use constant S_REC_NOHEADER     => 1;
use constant S_REC_NOFOOTER     => 2;
use constant S_REC_INVALID_TYPE => 3;
use constant S_REC_MALFORMED    => 4;
use constant S_REC_BAD_CS       => 5;
use constant S_REC_TOO_LARGE    => 6;
use constant S_REC_UNSORTED     => 7;
use constant S_REC_NO_FOOTER    => 8;
use constant S_REC_NO_HEADER    => 9;
use constant S_REC_UNKNOWN      =>10;

use constant MAX_BINARY_SIZE    => 0x100000; # This should cover a huge map

use constant S_REC_LENGTH       =>32;   # Max length of s recs generated
# s record format constants
our $first_line = "S00900004845414445524D";
our $last_line  = "S9030000FC";
our $s2 = "S2";
our $line_count=0;
our $bytes_read=0;            # For accounting of the number of bytes read
our $this_s_record="";        # Contains the S record currently under
                              # evaluation - used for error reporting

our $s_record_strict=1;       # Flag indicating the strictness of the S-record
                              # parsing process.  If true then one begin/end
                              # segment will be scanned for and no non-s2
                              # content will be acceptable between start and
                              # stop.
                              # If false, then anything that starts with S2
                              # will be examined and stored.  The begin/end
                              # records will be passively ignored, but will
                              # still set the state variables.

# Variables used for line to line accounting of S record processing.

our $addr_base=0;     # Base address to be subtracted from all S record
                      # addresses
our $firsttime_init;  # Control variable used in the parsing of the S records
our $target_offset=0; # The next offset in the binary buffer to be used.

our $debugging_s_rec=0; # Causes more output
our $errorlog=*STDERR;  # Set the errorlog stream

use constant S_REC_RTN => [
  "Successfully read S-Records",          # S_REC_SUCCESS
  "Could not find S-Record header",       # S_REC_NOHEADER
  "Could not find S-Record footer",       # S_REC_NOFOOTER
  "Bad S-Record type",                    # S_REC_INVALID_TYPE
  "Malformed S-Record.  Bad line length", # S_REC_MALFORMED
  "S-Record checksum failed",             # S_REC_BAD_CS
  "S-Record area too large",              # S_REC_TOO_LARGE
  "S-Records not properly sorted",        # S_REC_UNSORTED
  "S-Records have no footer",             # S_REC_NO_FOOTER
  "S_Records have no header",             # S_REC_NO_HEADER
  "S_Record parser returns unknown error" # S_REC_UNKNOWN
];

# #############################################################################
# #############################################################################
# ###
# ### Subroutines follow
# ###


sub process_line ($\@$)
# check to see if "S2" is at the beginning of the line and if the number of
# byte reported in the line matches the actual number of bytes in the line.
# It then performs a checksum verification.
# Returns nonzero if any of the tests fail.

{
  my( $line_len, $record_type, $num_bytes, $records);
  my $checksum=0;

  $_=$_[0];
  s/\r//g;                # decode_s_records should take care of this, but
  s/\n//g;                # To make this function callable standalone...
  my $line = $_;
  my $binary = $_[1];
  my $maxsize=0x10000;                  # default for max size
  $maxsize=$_[2] if (defined($_[2]));   # Redefine is size is specified

  my $index;              # Used for checksum and line store operations
  my $byte;               # Used for checksum and line store operations
  my $s_addr;             # The address of a line specified in the S rec

  $line_len = length( $line ) - 4;      # get length of line - rec type & num
                                        # bytes

  return (S_REC_SUCCESS) if ($line_len<0);
      # If the new line length is < 0, this is not any part of an S record
      # We passively accept it/ignore it.

  ($record_type, $num_bytes, $s_addr, $records)=unpack("A2A2A6A$line_len",$line);

# ###########################
# # Check S record type

  if($record_type ne $s2)
  {
    return (S_REC_INVALID_TYPE)
      if ($s_record_strict);  # Line didn't begin with s2

    return (S_REC_SUCCESS);   # If not in strict mode, just restart return
                              # gracefully after doing nothing.
  }
  # If we've made it here, the line begins with 'S2'

  $s_addr=hex($s_addr);       # Lets go ahead and convert the address

# ###########################
# # Check S record length

  $num_bytes = hex ($num_bytes);  # convert the ascii to it's numerical value

  return (S_REC_MALFORMED)     # Malformed S record if length mismatch
    if (2 * $num_bytes != $line_len);
  # If we've made it here, then we have an S2 record with a length that jives
  # with the specified length

# ###########################
# # Check S record chechsum

  for ($index=2;$index < ($line_len + 2); $index=$index+2)
  {
    $checksum=0xff&($checksum + hex(substr($line,$index,2)));    # Do the checksum
  }
  # As of now, we have summed up al the bytes in the line of S record data.

  # Lets test the checksum with the one specified in the S record.
  return (S_REC_BAD_CS)     # Failed checksum
    if ($checksum != (hex(substr ($line,$line_len + 2 ,2)) ^ 0xFF));

  # At a glance, it might seem sensible to combine the two similar 'for'
  # loops, but since the address is part of the checksum protected data,
  # we mustn't do anything with it until the checksum has been verified

  # If we've made it this far, then the line has a valid checksum.
  # Ok, if we're still alive by now, it's time to take the components from
  # the above work and store the data into a binary buffer

# ###########################
# # Compute target binary offset

  if ($firsttime_init)        # Handle the possibility that $addr_base
  {                           # is not yet initialized
    $addr_base=$s_addr;       # set the main base address to the first
                              # specified base address in the S records.
                              # This is where the rule requiring that the
                              # first S record must be the lowest address
                              # is based.
    $firsttime_init=0;        # Clear the firsttime flag
  }
  # Ok, now we simply placed the data into the buffer at the specified
  # the address:  $s_addr - $addr_base

# ###########################
# # Store each byte of the S record line into the binary buffer

  return (S_REC_SUCCESS) if ($bytes_read>=$maxsize);
  for ($index=10; $index < $line_len+2; $index=$index+2) # Parse through the
  {                                                      # data portion
    # Place the byte proportionately in the buffer
    $target_offset=($s_addr - $addr_base) + (($index - 10) / 2);
    return(S_REC_UNSORTED)   # We got an address lower than the first one
      if ($target_offset<0);

    return(S_REC_TOO_LARGE)   # We have a span that is too large
      if ($target_offset>MAX_BINARY_SIZE);

    # If we're here, then we can store the byte in the buffer
    printf ("Storing $target_offset %s\n", substr($line,$index,2)) if ($debugging_s_rec);

    $binary->[$target_offset] = hex(substr($line,$index,2));  # Store it!
    $bytes_read++;       # acounting...
    return (S_REC_SUCCESS) if ($bytes_read==$maxsize);
  }

  # Ok, That's about it.  let's jam
  return(S_REC_SUCCESS);
}


sub decode_s_records(\@\@$)
# Usage: decode_s_records(@s_record_src, @s_record_bin, $max_num_bytes)
# Processes the text in @s_record_src and stores the data in @s_record_bin.
# The S-Record specified starting address is placed in the $s_rec_base variable
# and the size of total span of addresses in the S block is placed in the
# $s_rec_size variable.  That is also the number of elements in @binary_dest.
{
  my $s_records=$_[0];                  # The specified source array regerence
  my $binary=$_[1];                     # The specified target array reference
  my $maxsize=0x10000;                  # default for max size

  $maxsize=$_[2] if (defined($_[2]));   # Redefine is size is specified

  my $s_rec_line;                       # Buffer to hold each line tested

  my $s_record_line;                    # The current line we're reading
  my $first_line_found = 0;             # Discovered beginning of S rec flag
  my $last_line_found = 0;              # Discovered end of S rec flag
  my $retval;

  $bytes_read=0;              # Reset this accounting variable
  $line_count=-1;             # Init line indicator to 0
  $firsttime_init=1;          # init global control variable to 0

  foreach $s_rec_line (@$s_records) # Iterate once for each element of the
  {                                 # array.
    $line_count++;
    $_=$s_records->[$line_count];
    s/\r//g;
    s/\n//g;            # remove newline
    tr/a-z/A-Z/;        # make upper case
    $this_s_record=$_;

# If we're still searching for the beginning
    if ($first_line_found == 0)          # first line has not been found
    {
      if ($this_s_record eq $first_line)
      {
        $first_line_found = 1;          # first line found
        next;                       # If we find the beginning, restart loop
      }
      # restart loop anyway if we are still pending for the beginning
      next if ($s_record_strict);   # if the $s_record_strict flag is true,
                                    # do not continue until we have gotton
                                    # the first_line sequence.  If there is
                                    # never a beginning line, then this
                                    # loop will never get past this point
                                    # at the end of the parsing, an error
                                    # is thus returned.
    }

    if($this_s_record eq $last_line)       # check for last line
    { # We've already found the first line, see if we have the last line
      $last_line_found = 1;           # last line found

      next;                     # Always restart if we hit the end record
    }
    elsif ((!$last_line_found)||(!$s_record_strict)) # Alright, we are between
                                                     # the first and last lines;
                                                     # let's boogie
    {
      if ($retval=process_line( $this_s_record, @$binary, $maxsize ))
      {
        return ($retval);       # Exiting on an error condition
      }
      # else the line was successfully processed
    }
  }

  return (S_REC_SUCCESS) if (!$s_record_strict);

  if ($first_line_found == 0)   # If we went through the whole buffer without
  {                             # finding the header of the S block
    return(S_REC_NO_HEADER);
  }
  elsif ($last_line_found == 0) # If we went through the whole buffer without
  {                             # finding the footer of the S block
    return(S_REC_NO_FOOTER);
  }
  return (S_REC_SUCCESS);       # Successful scan of file
}

sub generate_s_line(\@$$$)
# Usage:  generate_s_line ( @binary_buffer, $binary_buffer_offset, $length,
#         $s_addr_base )
# Generate an s record line from the binary buffer at a specified starting addr
# and to a specified length.  Do not try to generate an S2 record for more than
# 252 bytes (252 + 2 + 1 is the max count byte).  Generally, you will create
# records for no more than about 32 bytes.  This routine adds the global
# variable, s_base_addr to the $binary_buffer_offset.
# This function places a \n at the end of the generated line.
{
  my $binary_buffer=$_[0];
  my $offset=$_[1];
  my $length=$_[2];
  my $s_base_addr=$_[3];

  my $thebyte;

  # The length must be padded to account for the 3 bytes of address and 1 byte
  # of checksum.
  my $checksum = $length+4;     # The checksum includes the length
  my $outstring=sprintf ("S2%02X",$length+4);     # Start of the outstring

  my $start_addr=$s_base_addr+$offset;
  $outstring .= sprintf("%06X",$start_addr);
  $checksum += (($start_addr)+($start_addr>>8)+($start_addr>>16))&0xff;

  for (my $count=0;$count<$length;$count++)
  {  # Do this one time for each byte
    $thebyte = $binary_buffer->[$offset+$count];
    $outstring .= sprintf("%02X",$thebyte);
    $checksum += $thebyte;
  }
  $checksum = ($checksum & 0xff) ^ 0xff;
  $outstring .= sprintf ("%02X\n",$checksum);
  return $outstring;
}


sub generate_s_block(\@$$$)
# Usage:  generate_s_line ( @binary_buffer, $binary_buffer_offset, $length.
#         $s_addr_base )
# Generate an S record block for the binary buffer at a specified starting
# addr and to the specified length.  This function will produce lines for
# S_REC_LENGTH (32 bytes) each of the binary buffer and will include the
# start and stop records at the beginning and end of the block.

{
  my $binary_buffer=$_[0];
  my $offset=$_[1];
  my $length=$_[2];
  my $s_base_addr=$_[3];

  my $line_count=0;
  my $full_lines=$length/S_REC_LENGTH;    # Number of lines at S_REC_LENGTH/ea
  my $remainder=$length%S_REC_LENGTH;     # Number of bytes for the last line

  my @outbuffer;
  # Se divide the data into a number of segments of S_LENGTH, and then
  # generate a final line to take up the remainder.

  $outbuffer[$line_count++] = sprintf "$first_line\n";

  for ($line_count=1;($line_count)<$full_lines;$line_count++)
  {
    $outbuffer[$line_count]=generate_s_line( @$binary_buffer,
        ($line_count-1)*S_REC_LENGTH, S_REC_LENGTH, $s_base_addr );
  }
  # Ok, that's all the full lines.  Now for the final bytes
  if ($remainder)
  {
    $outbuffer[$line_count++]=generate_s_line( @$binary_buffer,
        ($line_count-1) * S_REC_LENGTH, $remainder, $s_base_addr );
  }
  else
  { # if no remainder, then we actually have 1 final whole line
    $outbuffer[$line_count++]=generate_s_line( @$binary_buffer,
	($line_count-1)* S_REC_LENGTH, S_REC_LENGTH, $s_base_addr );
  }

  $outbuffer[$line_count++] = sprintf "$last_line\n";

  return @outbuffer;
}
1;      # All modules must return a true value.
