#!/usr/bin/perl -w

# Kernel Extension Manager

# Revision history:
#   V1.0 - first version.  worked decently, but lacked interpackage dependency
#
#   V2.0 - major overhaul to support package interdependency and lots of other stuff
#   V2.01 - some minor bugfix I've forgotton about.
#
#   1.10.2005
#   V2.02 - Started maintaining revision history.  I simply added an error trap
#           for a !defined(@target_memory) case.  This fixes an odd behavior
#           when the .kemrc fails to set up a memory map.  The memory map is
#           normally set up by the schema files for the web interface.  This
#           fix is mainly intended to aid those of us who use the command line
#           interface.  The other prong of this fix is to be sure that each user
#           has basic memory map defined in their ~/.kemrc file.


# ##################### Main program runtime   ##############################
# ####
# ####

use diagnostics;

use Getopt::Std;
use Getopt::Long;
# commented out so this script will run on godaddy servers
# this is only used for 'interactive' mode which isn't used when customers
# use the online KEM
#use Term::ReadKey;

use POSIX;

use lib "./bin";                # Relative module location for web usage
use s_records 1.0;

use strict;   # Ok, no more variable declarations

package main;

# ############################################################################
# ############################################################################
# ############################################################################
# ##                                                                       ###
# #  B E G I N   O F    V A R I A B L E    D E C L A R A T I O N / I N I T  ##
# ##                                                                       ###
# ############################################################################
# ############################################################################
# ############################################################################

# ################### S Y S T E M   C O N F I G U R A T I O N ################
our $versiondate="2.3";     # This is printed in the startup banner and is
                            # for identification.
                            # private for internal debugging only

our $eol="\r\n";            # this is placed at the end of each line
our $verbose_mode=0;        # Set by -v flag
our $debug_list=0;          # print extra info while generating the list
our $debug_relocation=0;
our $debug_dependency=0;    # Dependency code debugging

our $lowlevel_cmdline=0;    # Debugging flag

our $errorstream=*STDOUT;   # We set this to stdout initially

our $s_rec_dest_xaddr=0x010000;         # the address used for the binary
                                        # image before moving it to flash
our @target_memory=();            # this list is initialized by the config file
our $code_target_index=0;         # indices into @target_memory
our  $var_target_index=0;

our $show_all=0; # flag indicating whether or not all packages should be shown
                 # regardless of their visibility status

# These are the filenames for the system pre/post ambles.
# individual packages may also have pre/post ambles for the install file.
our $install_pre_filename="install_preamble.txt";
our $install_post_filename="install_postamble.txt";

our $forth_pre_filename="forth_preamble.4th";
our $forth_post_filename="forth_postamble.4th";

# packages don't have pre/post ambles here since they comprise the middle
# part.

our $sys_c_src_pre_filename="c_src_preamble.c";
our $sys_c_src_post_filename="c_src_postamble.c";

our $sys_c_hdr_pre_filename="c_hdr_preamble.h";
our $sys_c_hdr_post_filename="c_hdr_postamble.h";

our $sys_readme_pre_filename="readme_pre.txt";    # Prepended to readme.txt
our $sys_readme_post_filename="readme_post.txt";  # Appended to readme.txt

our $copy_directive='to.flash not iftrue ." FLASH FAIL!" beep endiftrue';

# Declare the main global variables
our @install_file_buffer;   # This array will become the install file
our @forth_file_buffer;     # This array will become the forth file

our @c_source_file_buffer;  # This array will become the c source file
our @c_header_file_buffer;  # This array will become the c header file
our @readme_file_buffer;    # This array will become the readme file


# our @forth_symbols_buffer;  # Forth symbols buffer
our @c_symbols_buffer;      # C symbols buffer

our @allocation_report;     # array of lines of text containing pkg name,
                            # version, and location
push @allocation_report,"----Beginning of package memory allocation listing----";
our @memory_diagram;


# ####################################################################
# ####
# ####    Filestreams data structure.
# ####

  # The hash below contains all of the filestreams for the different output
  # files that the kem generates.
  our %filestreams = (
    'install_file'            => 0,   # package install file (.txt)
    'forth_file'              => 0,   # Forth download file (.4th)
    'c_source_file'           => 0,   # C source file (.c)
    'c_header_file'           => 0,   # C header file (.h)
    'readme_file'             => 0    # kem readme file (.txt)
    );

  our %filenames = (
    'install_file'            => "install.txt",   # package install file (.txt)
    'forth_file'              => "library.4th",   # Forth download file (.4th)
    'c_source_file'           => "library.c",   # C source file (.c)
    'c_header_file'           => "library.h",   # C header file (.h)
    'readme_file'             => "readme.txt"    # kem readme file (.txt)
    );


our $command;               # command line in argv[0] (list/build)

# This default may be overwritten
our $pkg_source_path="./";           # source path for the packages being read

our $outpath="./kemout/";   # The root path for the various output files
                            # This variable is set either by the command
                            # line or based on the current working
                            # directory

our $quiet;                 # this flag is set by the command line args
                            # and determines whether status reporting info
                            # be written to stdout.  With this flag set
                            # false, the progress of the program is
                            # written to stdout.  Errors are always
                            # written to stderr.

our %cmd_options;           # a hash of all the possible options.  This hash
                            # contains options' values indexed by the option
                            # names.

  # ####################################################################
  # ####
  # ####    Option data structure definitions
  # ####

  our @required_options = (
    'kem_version_required',
    'package_name',
    'readmefile',
    'webinfofile',
    'c_ct_init',
    'forth_ct_init',
    'code_granularity',
    'variable_granularity',
    'dp_page',
    'dp_addr',
    'np_addr',
    'np_page',
    'vp_addr',
    'vp_page'
    );

  # This is the list of options that will be placed in the kernel extension
  # description section of the ini file.  All the keys listed below must
  # exist in the options hash.
   our %allowed_options = (
     'kem_version_required',      '',
     'package_description',       '',
     'package_version',           '',
     'package_name',              '',
     'package_date',              '',
     'c_ct_init',                 '',
     'forth_ct_init',             '',
     'code_granularity',          '',
     'variable_granularity',      '',
     'dp_addr',                   '',
     'dp_page',                   '',
     'np_addr',                   '',
     'np_page',                   '',
     'vp_addr',                   '',
     'vp_page',                   '',
     'c_symbolfile',              '',
     'c_sourcefile',              '',
     'c_headerfile',              '',
     'readmefile',                '',
     'keg_reportfile',            '',
     'raw_forthfile',             '',
     'extension_binfile',         '',
     'extension_sfile',           '',
     'package_inifile',           '',
     'readmefile',                '',
     'visible',                   '1',
     'webinfofile',               ''
     );

our @ini_listing;             # This array contains a list of the directories
                              # in the package source path


# Packages hash array:  This one is important.  This is an array of hashes
# with one element per package scanned.  This is not the list of packages
# to be built, but a complete list of all packages found.  Each anonymous
# hash in this array corresponds to the options specified in the ini file
# for a particular hash.

our @packages; # =({},{},{});
our %pkglist;       # hash used for easy testing of valid package names
our %installed_pkgs;
our @installed_pkg_order;   # To be populated in resolve_dependencies()

# ********************************************************
# ********************************************************
# ***
# ***   Multi target management.  This is new in version 2.0
# ***     We use arrays that can be of variable length to contain
# ***     the various target areas.  The packages intended for
# ***     each area are selected ahead of time, and then written to
# ***

# To allocate memory, all packages are scanned, and when a dependency
# is noted, the dependency is added to the package list if it is not
# already present.  When a local dependency is encountered, it is moved
# to the front of the line and added with the package that depended on it.
# At that mount, all other packages are checked for that dependency.
# Any found will be added next.  If it doesn't fit, then that package
# is placed on the next page, and a copy of the dependency is added there.
# nonlocal dependencies are added normally, and given no special treatment.

# An optimization is that all packages are scanned, and local dependencies
# are scanned for.  local dependencies are added first ranked by the
# number of other packages that depend on them.  Once the highest ranking
# local dependency is added, the packages that locally depend on it are
# added next.  If they cannot all fit on a page, then the next section
# is started with a copy of that local dependency.  At that point,
# another dependency is added.

our $total_vp_used=0;       # Used for accounting with regard to setting the
                            # beginning vp area of the library files
our $next_free_vp=0;        # Maintained by allocate_packages to be used
                            # in library files as the start of the customer VP.
# ####################################################################
# ####
# ####    Prompt text declarations
# ####

  our $prog_id_text=   <<ENDPROGID;
  Kernel Extension Manager (kem) $versiondate
ENDPROGID

  our $usage_info_text=<<ENDUSAGE;
  Usage:
    keg detail_list  [OPTIONS]
    keg list  [OPTIONS]
    keg build [OPTIONS] pkg1, pkg2, ...
    keg list_symbols pkg1, pkg2, ...  [OPTIONS]
ENDUSAGE

  our $help_text=  <<ENDOFHELP;

  Available universal options:
  -v, --verbose   Do everything in a verbose fashion.  The extra information
                  will be sent to stdout.
  -V, --version   Print version info and exit.
  -q, --quiet     Quiet.  Don\'t print anything to stdout.
  -h, --help      Print this help and exit.
  -s <path>  Specify a different path in which to read packages

  <no command>: Interactive mode

  list:
    Produces a summary list of all valid kernel extension packages in the
    source path.

  detail_list:
    Produces a detailed list of all valid kernel extension packages including
    their dependency files and the web info filename.  This mode was intended
    to be used by the web CGI scripts for easy parseability, but it is
    generally useful.

  build:
    Builds the library files and the install file for a customized kernel
    extension set.  This is the main function of this program
    -b <Begin xaddress in hex>   The starting xaddr for the extensions
    -o <path>  Specify a different path to which package files are written
  list_symbols:
    This is really just meant for the keg to use as a convinience.  This is
    Just a raw list of each symbol for which create statements will be issued.
    This is used by the keg to generate a fake symbol list when compiling
    packages that need to be able to make kep calls.
ENDOFHELP

# ############################################################################
# ############################################################################
# ####
# ####    Subroutine declaration
sub eprint(@);
sub error(@);
sub crunch_cmd_options();

sub load_pkg_info();
sub build_packages();
sub link_packages();
sub allocate_packages();
sub check_dependencies(@);
sub add_package($);

sub generate_memory_diagram();
sub resolve_dependencies();
# Streams

sub write_files();
sub write_symlist();
sub close_outfiles();
sub open_outfile($);
sub read_description(\%$);
sub list_ini_sections($);
sub read_ini_section(\%$$);
sub get_dirlist();
# ############################################################################
# ############################################################################
# ############################################################################
# #####                                                                    ###
# ####  E N D    O F    V A R I A B L E    D E C L A R A T I O N / I N I T  ##
# #####                                                                    ###
# ############################################################################
# ############################################################################
# ############################################################################



# For reasons unknown to me, this routine only works when it is up here.
# If I put with all the other subroutines, perl will complain about
# uninitialized values.

sub do_interactive()
{
  format pkgreportmenu_TOP =
ID      Name                     Version Date
-----------------------------------------------------------------------------------
.
  $~="pkgreportmenu_TOP";
  write;
  my $pkgnum=0;
  foreach my $this_element (@packages)
  {
    format pkgreportmenu =
@<< --- @<<<<<<<<<<<<<<<<<<<<<<< @<<<<<  @<<<<<<<<<<<<
$pkgnum, $this_element->{'package_name'}, $this_element->{'package_version'}, $this_element->{'package_date'}
.
    $~="pkgreportmenu";
    write;      # generate the report
    $pkgnum++;
  }
  print "List package numbers separated by spaces or commas: ";
  my $answer = ReadLine(0);
  my @pkgs_to_build=split(/[ ,]+/,$answer);
  my $cmd_line="kem build -v ";
  foreach my $this_one (@pkgs_to_build)
  {
    $cmd_line.=$packages[$this_one]->{'package_name'}."_v".$packages[$this_one]->{'package_version'}." ";
  }
  print "About to build your selections with the following command line:\n";
  print $cmd_line."\n";
  print "Proceed?[Yn]";
  my $inkey;
  ReadMode (4);   # enable raw mode (so that no cr/lf is required to answer the question)
  while (!defined($inkey=ReadKey(-1))) { }
  ReadMode (0);   # back to normal mode
  if ((lc($inkey) eq "y")||($inkey eq "\n"))
  {
    print "  Proceeding..\n";
    system("$cmd_line");
  }
  else
  {
    print "Aborting\n";
    exit(0);
  }


}



sub write_text_summary()
{
  # Gernerates a nicely formatted summary of all available kernel extenstions.

  format pkgreport_TOP =
Name            Description                                 Version Date
-------------------------------------------------------------------------------
.
  my $this_element;
  foreach $this_element (@packages)
  {
    if (defined($this_element->{'visible'}))
    {
      next if ((($this_element->{'visible'} eq '0')||($this_element->{'visible'} eq 'false')) &&
        !$show_all);
    }
    format pkgreport =
@<<<<<<<<<<<<<<< @<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< @<<<<<  @<<<<<<<<<<
$this_element->{'package_name'},  $this_element->{'package_description'}, $this_element->{'package_version'}, $this_element->{'package_date'}
.
    $~="pkgreport";
    write;      # generate the report
  }
}

sub write_text_detail()
{
  # Gernerates a nicely formatted summary of all available kernel extenstions.

  print "Detailed Summary of packages.  Formatted for easy parsing by the CGI Scripts\n";
  print "---begin listing---\n";   # This line is searched for by the cgi
  my $this_element;
  foreach $this_element (@packages)
  {
    if (defined($this_element->{'visible'}))
    {
      next if ((($this_element->{'visible'} eq '0')||($this_element->{'visible'} eq 'false')) &&
        !$show_all);
    }
    printf "Package Name:  %s\n", $this_element->{'package_name'};
    printf "Package Description:  %s\n", $this_element->{'package_description'};
    printf "Package Version:  %s\n", $this_element->{'package_version'};
    printf "Package Date:  %s\n", $this_element->{'package_date'};
    printf "Web Info File:  %s\n", $this_element->{'base_path'}.$this_element->{'webinfofile'};
    print "\n";
  }
  print "---end listing---\n";   # This line is searched for by the cgi
}



# ############################################################################
# ############################################################################
# ############################################################################
# ######                                                             #########
# #####  B E G I N N I N G    O F    R U N T I M E                    ########
# ######                                                             #########
# ############################################################################
# ############################################################################
# ############################################################################


#print $versiondate;
$|=1;                         # Turn on autoflushing of stdout
crunch_cmd_options();         # Reads cmd line options and config files

# NEwly added error trap
if (!@target_memory)
{
  eprint "No target_memory blocks specified.  Check your .kemrc file.\n";
  exit;
}

if ($command eq "interactive")
{
  my $num_pkgs=load_pkg_info();
  do_interactive();
  exit(0);          # interactive mode always exits
}
elsif ($command eq "list")
{
  print "Executing command: list\n" if ($verbose_mode);
  my $num_pkgs=load_pkg_info();

  # Do the rest of the list command
  # ok.  We have our hash, all we have to do is summarize it in
  # some reasonable fashion and we're done!
  print "\nFound $num_pkgs packages\n";
  print "Summary of available Kernel Extension Packages\n";
  write_text_summary();    # Produce the output
  print "\n";
}
elsif ($command eq "detail_list")
{
  print "Executing command: detail_list\n" if ($verbose_mode);
  my $num_pkgs=load_pkg_info();

  # Do the rest of the detail_list command
  # ok.  We have our hash, all we have to do is summarize it in
  # some reasonable fashion and we're done!
  print "\nFound $num_pkgs packages\n";
  print "Detailed summary of available Kernel Extension Packages\n";
  write_text_detail();    # Produce the output
  print "\n";
}
elsif ($command eq "build")
{
  my $num_pkgs=load_pkg_info();
  if ($verbose_mode)
  {
    print "Executing command: build\n";
    print "Building kernel extension library\n";
    print "Validating requested extensions: \n";
  }
  if (!scalar(@ARGV))
  {
    eprint "Nothing to do!  No package names specified.\n";
    exit(1);
  }
  foreach (@ARGV)
  { # Iterate through each requested extension

    # Code to figure out version numbers BEGIN
    my $pkg_name=$_;
    if ($pkg_name =~ /^(.*)_v[0-9]+\.[0-9]+[[:alpha:]]?$/) # Allow lowercase
    {                                                    # sub-versions
      print "Full ver info specified for $pkg_name\n";
      # $_=$1;
    }
    else
    {   # We must figure out the version ourselves
      print "Filling in highest version number for package $pkg_name\n"
          if ($verbose_mode);

      my $highest_version="0.0";    # to have the highest version number after the
                                  # loop
      my @verslist;

      foreach my $this_package (@packages)
      {
        # iterate through each package and find out what versions are available
        if ($this_package->{'package_name'} eq $pkg_name)
        {
          push @verslist, $this_package->{'package_version'};
# Replaced this with alphanumeric sort
#            $highest_version = $verslist[0];
#            $highest_version = $this_package->{'package_version'}
#                if ( strtod($this_package->{'package_version'}) >
#                     strtod($highest_version) );
        }
      }
      @verslist=sort(@verslist);
      if($verbose_mode)
      {
        print "Versions possible";
        foreach my $this_version (@verslist)
        {
          print $this_version." ";
        }
        print "\n";
      }

      $highest_version=$verslist[-1];
      printf "Version chosen: %s\n",$highest_version if ($verbose_mode);
      $pkg_name .= '_v'.$highest_version;       # Add version to name
    }
    # Code to figure out version numbers END

    if (!defined($pkglist{$pkg_name}))
    {
      eprint "Couldn't find $_ in the package search path\n";
      exit(1);
    }
    print "      $pkg_name\ " if ($verbose_mode);

    add_package($pkg_name);
    print "..Success!\n" if ($verbose_mode);
  }
  allocate_packages();  # parse packages and dependencies and choose
                        # locations for each package.

  build_packages();     # This is where most of the magic is
  link_packages();      # Plug the packages into each other (dependencies)
  write_files();        # This function does all of the output file writing
}
elsif ($command eq "list_symbols")
{
  my $num_pkgs=load_pkg_info();
  if ($verbose_mode)
  {
    print "Executing command: list_symbols\n";
    print "Preparing a list of symbols\n";
    print "Validating requested extensions: \n";
  }
  if (!scalar(@ARGV))
  {
    eprint "Nothing to do!  No package names specified.\n";
    exit(1);
  }
  foreach (@ARGV)
  { # Iterate through each requested extension

    # Code to figure out version numbers BEGIN
    my $pkg_name=$_;
    if ($pkg_name =~ /^(.*)_v[0-9]+\.[0-9]+[[:alpha:]]?$/)
    {
      print "Full ver info specified for $pkg_name\n";  #
    }
    else
    {   # We must figure out the version ourselves
      print "Filling in highest version number for package $pkg_name\n"
          if ($verbose_mode);

      my $highest_version=0.0;    # to have the highest version number after the
                              # loop
      foreach my $this_package (@packages)
      {
        # itterate through each package and find out what versions are available
        if ($this_package->{'package_name'} eq $pkg_name)
        {
          $highest_version = $this_package->{'package_version'}
              if ( strtod($this_package->{'package_version'}) >
                   strtod($highest_version) );
        }
      }
      printf "Version chosen: %s\n",$highest_version if ($verbose_mode);
      $pkg_name .= '_v'.$highest_version;       # Add version to name
    }
    # Code to figure out version numbers END
    if (!defined($pkglist{$pkg_name}))
    {
      eprint "Couldn't find $_ in the package search path\n";
      exit(1);
    }
    print "      $pkg_name\n" if ($verbose_mode);

    add_package($pkg_name);               # The new way
    print "..Success!\n" if ($verbose_mode);
  }
  # Write_symlist will produce a parsable list of the
  # forth callable symbols in each of those packages.  This functionality
  # is for use by the keg.
  write_symlist();      # write a list of the symbols for the ked
}
else
{
  eprint "Invalid command: $command\n";
  if ($command=~/^-/)
  {
    eprint "It looks like you might have forgotton to specify a command alltogether\n";
    eprint "as a switch was found where a command was expected.\n";
  }
  exit(1);
}

print "Kernel Extension Manager exiting normally\n" if ($verbose_mode);

exit(0);    # this exit is kinda pointless, but it ensured finality

# ############################################################################
# ############################################################################
# ############################################################################
# ######                                                             #########
# #####  E N D    O F    R U N T I M E   --   S U B S    F O L L O W  ########
# ######                                                             #########
# ############################################################################
# ############################################################################
# ############################################################################


# ######################   End block - what happens on any exit    ###########
# ####
# ####

END {
  print "Closing outfiles...\n" if ($verbose_mode);
  close_outfiles();
  print scalar(localtime),"  - KEM Exiting..\n" if (!$quiet);
}






#our %c_func_hash;
#our %forth_func_hash;
sub add_package($)
{
  my $this_package=$_[0];
  if (!defined($installed_pkgs{$this_package}))
  {
    # The following line creates an anonymous hash to which
    # $installed_pkgs{$this_package} is a reference
    $installed_pkgs{$this_package}={};
    print "Preprocessing package: $this_package\n";

    # Copy each entry of the installed_pkgs hash.
    foreach (keys %{$packages[$pkglist{$this_package}]})
    {   # Copy each entry into the installed_pkgs hash.  Makes for cleaner
        # code later on.
      $installed_pkgs{$this_package}{ini}{ked}{$_} =
          $packages[$pkglist{$this_package}]{$_};
      # print"gonna add $_\n";
    }

    my $inipath =
      $installed_pkgs{$this_package}{ini}{ked}{base_path} .
      $installed_pkgs{$this_package}{ini}{ked}{package_inifile};

    print "Reading memory usage information: $this_package\n" if ($verbose_mode);
    # We need to get more information from the ini file.  Populate the memory section
    if (read_ini_section( %{ $installed_pkgs{$this_package}{ini}{memory}}, "[memory]", $inipath))
    {
      eprintf ("Failed reading memory section from inifile:$inipath\n");
      exit(1);
    }

    print "Reading dependency mapping information: $this_package\n" if ($verbose_mode);
    # Read the Forth function addresses -- this is for dependency resolution later on.
    if (read_ini_section( %{ $installed_pkgs{$this_package}{ini}{all_functions}}, "[all_functions]", $inipath))
    {
      eprintf ("Failed reading all_functions section from inifile:$inipath\n");
      exit(1);
    }

    print "Reading dependencies: $this_package\n" if ($verbose_mode);
    # Populate dependency section of main hash
    if (read_ini_section ( %{ $installed_pkgs{$this_package}{ini}{kep_symbols}}, "[kep_symbols]",$inipath))
    {
      eprintf ("Failed reading kep_symbols section from inifile:$inipath\n");
      exit(1);
    }
  }
  else
  {     # Do nothing if it has already been added
    print "Package $this_package already present.  Ignoring...\n";
  }
}

sub load_pkg_info()
{
  # This function scans the $pkg_source_path for ini files and loads the
  # $packages array with hashes.  This function returns the number of
  # packages that it found.

  print "Scanning package path: $pkg_source_path\n" if ($verbose_mode);
  get_dirlist();      # populate the list of inifiles
  my $inifile;
  my $fullinipath;
  my $pkg_index=0;    # Index counter for the hash array
  foreach $inifile (@ini_listing)
  { # Iterate through all ini files, selected or not
    print "Scanning package: $inifile\n" if ($verbose_mode);

    $fullinipath=$pkg_source_path.$inifile;

    # The odd looking construction below is an on-the-fly creation
    # on an anonymous hash whose reference is passed into the function.
    read_description( %{ $packages[$pkg_index] } , $fullinipath);

    # /\/\/ May need to put some kind of sanity check in here to
    my $fail=0;
    my $this_key;
    foreach (@required_options)
    {  # Parse through each required option and make sure it is defined {
      #print "Testing $_\n";
      next if (defined($packages[$pkg_index]->{$_}));

      # If we're still here, then we hit an undefined value
      eprintf ("The required option, $_, was missing from this ini file:%s\n",$fullinipath);
      eprint "Skipping this package\n";
      $fail=1;
      last;
    }
    if (!$fail)
    {
      # This is just a list of packages that is maintained
      $pkglist{$packages[$pkg_index]
          ->{'package_name'}.'_v'.$packages[$pkg_index]->{'package_version'}}
          = $pkg_index;
      $pkg_index++;
    }
    else
    {
      pop @packages;  # Remove the last element from the array
    }
  }
  return($pkg_index);
}


sub read_description(\%$)
# Usage: read_description(%options_hash, $fullfilename)
# Read the description information for a package and puts it into the hash
# reference provided.  If the file is unreadable, returns nonzero.  Otherwise
# returns zero.
{
  my $fh;
  my $hashref=$_[0];    # reference to the options hash
  my $inifile=$_[1];    # the inifile name
  my $got_kep_symbols=0;
  my $got_ked=0;
  if (!open($fh,"<$inifile"))
  {
    return(1);    # Couldn't open the file, so let's quietly exit
  }
  # Ok, We have successfully opened the ini file.


  while (<$fh>)
  {
    # Iterate through each line of the file
    s/\r|\n//g;   # get rid of any eol char
    s/\s+$//;     # Get rid of any trailing spaces
    if ($_ =~ /^\[/)
    {             # This is the universal way of turning everything
                  # off since we are beginning a new section.
      $got_ked=0;
      $got_kep_symbols=0;
    }
    if (lc($_) eq "[kernel extension description]")
    {
      $got_ked=1;   # Mark the state variable indicating that we have the
                    # beginning of the kernel extension description section
      next;         # No need to hang around though
    }


    if ($got_ked)
    {
      # We are in the ked section, so let's populate the hash
      # Since we wanna be a bit controlled about things, we only
      # record keys that are in a list of permitted keys rather than
      # making new keys of anything in the inifile

      # print "line: $_\n";

      # first, let's get a pair of key/value
      my ($key,$value);
      if ((($key,$value)=split('='))==2)
      {   # We have a valid key/value pair
        if (defined($allowed_options{$key}))
        {   # This check allows us to filter the keys allowed
          print "Got a standard option: $key=$value\n" if ($debug_list);
          # Now the have validated the option, we need to add it to an
          # array of anonymous hashes.
          $hashref->{$key}=$value;    # It places the lotion in the basket
        }
        else
        {
          print "Ignoring the invalid option has been specified: $key\n"
            if ($debug_list);
        }
      }
    }
  }
  close($fh);

  # We need to add a member to the packages array hash for the full filepath
  # with trailing '/'.  This will be used for all other specified files in the
  # inifile.  Note that we do this after the file has been loaded.
  $hashref->{'base_path'} = $inifile;
  $hashref->{'base_path'} =~ s/\/[^\/]+$/\//; # sheesh! In normal terms, chop
                                              # off all but the name of the
                                              # ini file, and leave the
                                              # trailing / on the path.
  $hashref->{'top_path'} = $hashref->{'base_path'};
  $hashref->{'top_path'} =~ /(\/[^\/]+\/)$/;
  $hashref->{'top_path'} = $1;  # Get only the last directory level
  return (0) if (!$got_ked);    # This will happen if we went through the
                                # whole file and didn't find the ked section
  # Ok, we're still here; that means we at least found the section in the ini
  # file.  Now, let's see if we have gotton all the required pieces of data

  return (1);
}



sub read_ini_section(\%$$)
# Usage: read_description(%options_hash, %validity_hash, $section_name, $fullfilename)
# Read the description information for a package and puts it into the hash
# reference provided.  If the file is unreadable, returns nonzero.  Otherwise
# returns zero.
{
  my $fh;
  my $hashref=$_[0];        # reference to the options hash
 # my $validity_hash=$_[1];  # Only options whose names are listed here will be
  my $section_name=$_[1];   # read name of the section (including the brackets)
  my $inifile=$_[2];        # the full inifile name
  print "Reading $inifile for section $section_name\n" if ($verbose_mode);
  my $got_it=0;
  if (!open($fh,"<$inifile"))
  {
    return(1);    # Couldn't open the file, so let's quietly exit
  }
  # Ok, We have successfully opened the ini file.


  while (<$fh>)
  {
    # Iterate through each line of the file
    s/\r|\n//g;   # get rid of any eol char
    s/\s+$//;     # Get rid of any trailing spaces
    if (lc($_) eq $section_name)
    {                                                   # after fixing keg!!!
      $got_it=1;    # Mark the state variable indicating that we have the
                    # beginning of the kernel extension description section
      next;         # No need to hang around though
    }
    if ($got_it)
    {
      # We are in the right section, so let's populate the hash
      # Since we wanna be a bit controlled about things, we only
      # record keys that are in a list of permitted keys rather than
      # making new keys of anything in the inifile

       # print "line: $_\n";

      if ($_ =~ /^\[/)  # trap the beginning of the next section
      {
        # We have reached the end of the ked section
        # print "At end of section.\n";
        last;
      }
      # first, let's get a pair of key/value
      my ($key,$value);
#      if (~/(.+)=([^= ]+$)/)
#      {
#        print "\n\nDEBUG:::: $1 - $2\n\n";
#      }
      # if ((($key,$value)=split('='))==2)
      if (m/(.+)=([^= ]+$)/)
    #  if (($1 ne "") && ($2 ne ""))
      {   # We have a valid key/value pair
          #($key,$value)=($1,$2);
          $key=$1;
          $value=$2;
#           print "\n\nDEBUG:::: $key - $value\n\n";
           $hashref->{$key}=$value;
      }
    }
  }
  close($fh);

  return (1) if (!$got_it);    # This will happen if we went through the
                                # whole file and didn't find the ked section
  # Ok, we're still here; that means we at least found the section in the ini
  # file.  Now, let's see if we have gotton all the required pieces of data

  return (0);
}



sub list_ini_sections($)
# Usage: read_ini_sections( $fullfilename)
# Read the description information for a package and puts it into the hash
# reference provided.  If the file is unreadable, returns nonzero.  Otherwise
# returns zero.
{
  my $fh;
  my @sections;
  my $inifile=$_[0];        # the full inifile name
  print "Reading $inifile to get sections list\n" if ($verbose_mode);
  my $got_it=0;
  if (!open($fh,"<$inifile"))
  {
    return(1);    # Couldn't open the file, so let's quietly exit
  }
  # Ok, We have successfully opened the ini file.


  while (<$fh>)
  {
    # Iterate through each line of the file
    s/\r|\n//g;   # get rid of any eol char
    s/\s+$//;     # Get rid of any trailing spaces
    if (lc($_)=~/^\[(.+)\]/)
    {   # we have a section tag
      push @sections,$1;
    }
  }
  close($fh);

  return (@sections);
}



sub check_dependencies(@)
{
  # This function parses the list it is given and returns a list of unmet
  # dependencies.  Each item in the list must already be initialized
  # in the %installed_pkgs hash.  This function fills in the kep_symbols
  # section and returns any requested dependencies not already listed
  # in %installed_pkgs.  By calling this in a loop that does not terminate
  # until all dependencies have been met, dependencies may be recursive.

  my @pkgs=@_;              # get the list of packages we were provided with
  my @unmet_dependencies;   # This is what we'll return

  foreach my $this_package (@pkgs)
  {
    # iterate through the dependency listings
    foreach my $dep_key (keys %{$installed_pkgs{$this_package}{ini}{kep_symbols}} )
    {
      if ($dep_key=~/^kep_far_dependency_[[:digit:]]+/)
      {
        # In here, we get to look at the dependencies of each package
        print "looking at $dep_key = $installed_pkgs{$this_package}{ini}{kep_symbols}{$dep_key}\n"
          if ($debug_dependency);

        # before we add this to the unmet dependency list, we
        # check it against the %installed_pkgs list.

        my $this_dep=$installed_pkgs{$this_package}{ini}{kep_symbols}{$dep_key};
        if (!defined($installed_pkgs{$this_dep}))
        {
          # Add only if not already listed
          push (@unmet_dependencies,$installed_pkgs{$this_package}{ini}{kep_symbols}{$dep_key});
        }
        else
        {
          print "Found dependency, $this_dep, that has already been explicitly requested.\n"
            if ($debug_dependency);
        }
      }
    }
  }
  return(@unmet_dependencies);
}


sub resolve_dependencies()
{
  # This function contains a loop that cycles through first the initially
  # requested packages, and generates a list of all dependencies not already
  # requested.  That list is then parsed in the same way, and any new dependencies
  # discovered on that level will be added to the installed packages hash.  This
  # process ends when all dependencies are present in %installed_pkgs.  Thus, dependency
  # nesting may be infinite.

  print "Processing dependencies\n" if ($verbose_mode);
  my @list=check_dependencies(keys (%installed_pkgs));
  while (@list)
  {   # Iterate through the recently generated dependency list adding any additional
      # packages
    foreach (@list)
    { # iterate through the current version of the list to error check
      if (!defined($pkglist{$_}))
      {
        eprintf ("Bad dependency request.  $_ doesn't seem to exist.\n");
        exit(101);    # bail here
      }
      print "Automatically adding dependency $_ which was not explicitly requested.\n"
        if ($debug_dependency);
      add_package($_);    # Add each package in the list
    }
    # At this point, the packages in @list have been frisked and added to
    # the %installed_pkgs hash
    @list=check_dependencies(@list);    # process a level of dependencies
    # The new list returned will contain any dependencies discovered in
    # this iteration that now need to be scanned.  Those will cause
    # this while loop to repeat.
  }

  # Nick 20150224 - As an improvement to space efficiency and determinism,
  #                 sort packages by descending code size.
  # An optimal method would involve making a list of all combinations of remaining packages
  # at the beginning of each region sorted by size of that combination, and choose the one
  # that best utilizes the region (largest combination of smaller size than the region).
  @installed_pkg_order =
    sort { hex($installed_pkgs{$b}{ini}{memory}{binary_size}) <=>
           hex($installed_pkgs{$a}{ini}{memory}{binary_size}) } keys %installed_pkgs;

  print "Finished processing dependencies\n" if ($verbose_mode);
}



sub memory_diagram_line
{
  use integer;
  die if ( @_ != 3 );   # Used, Size, Bytes/Block
  my $line = 'X' x ( ( $_[0] + $_[2] / 2 ) / $_[2] );
  $line .= 'x' if ( $_[0] && ( $_[0] - 1 ) % $_[2] < $_[2] / 2 );
  $line .= '.' x ( ( $_[1] + $_[2] - 1 ) / $_[2] - length( $line ) );
  return( $line );
}

sub generate_memory_diagram()
# This function writes a text based graphical diagram of the memory usage
# of all the packages, showing all memory areas that are declared as target_memory
{
  push @memory_diagram,"";
  push @memory_diagram,"----Beginning of memory usage diagram----";
  my $c=0;
  my $divisor=0x100;
  push @memory_diagram,sprintf ("** Vars usage each block is 0x%x bytes.  X=Used x=partially used .=not used",$divisor);
  foreach (@target_memory)    #
  {
    next if ($_->{'type'} ne 'variable');
    push @memory_diagram,sprintf "Vars %02d: Range: %06x - %06x Used:%04x [%s]",
      $c, $_->{'start_ptr'}, $_->{'end_ptr'}, $_->{'used'},
      memory_diagram_line( $_->{'used'}, $_->{'size'}, $divisor );
    $c++;
  }
  $c=0;
  $divisor=0x400;
  push @memory_diagram,"";
  push @memory_diagram,sprintf ("** Code usage each block is 0x%x bytes.  X=Used x=partially used .=not used",$divisor);
  foreach (@target_memory)    #
  {
    next if ($_->{'type'} ne 'code');
    push @memory_diagram,sprintf "Code %02d: Range: %06x - %06x Used:%04x [%s]",
      $c, $_->{'start_ptr'}, $_->{'end_ptr'}, $_->{'used'},
      memory_diagram_line( $_->{'used'}, $_->{'size'}, $divisor );
    $c++;
  }
  push @memory_diagram,"----End of memory usage diagram----";

}

# ***************
#   allocate_packages()
#     This function parses the list of requested packages (in %installed_pkgs)
#   and expands the dependencies and allocates space in the target memory
#   areas needed to hold all requested packages and their dependencies.

    # For each pass of the main loop, here is the todo list:
    #   -load the memory part of the %installed_pkgs hash
    # ---   Code  pointer   ---
    #   -adjust the current pointer of the current area to meet granularity
    #   -add total code size to current pointer and check for overflow
    #       - if overflow, then we must repeat with the next section
    #   -write variables into the packages target area indicating placement
    #     intentions

    # --- Variables pointer ---
    #   -adjust the current pointer of the current area to meet granularity
    #   -add total code size to current pointer and check for overflow
    #       - if overflow, then we must repeat with the next section
    #   -write variables into the packages target area indicating placement
    #     intentions
    #
sub allocate_packages()
{
  # This routine is where the locations of all the packages are chosen.
  # It does dependency sorting and places all packages in the memory
  # sections.  This routine is new as of 2.0

  resolve_dependencies();   # calls add_package on any dependencies
  # at this point, all the vital package information should already
  # be loaded into the %packages hash, and the %installed_pkgs
  # hash has only been initialized and contains the [kep_symbols]
  # keys.

  # NAMING CONVENTION: To mitigate my own confusion, references beginning with
  # target relate to the output target memory hash.  references beginning with
  # pkg refer to the %installed_pkgs or @packages hash.
  my $last_target=scalar(@target_memory);

  foreach  (@target_memory)
  { # Do all preliminary target_memory list initialization here.
    $_->{'cur_ptr'}=$_->{'start_ptr'};    # Set each areas current ptr variable to the start
    $_->{'size'}=$_->{'end_ptr'} - $_->{'start_ptr'};
    $_->{'used'}=$_->{'cur_ptr'} - $_->{'start_ptr'};
  }

  # **********************************************************************
  # **********************************************************************
  # ***                                            *********
  # ***   Beginning of per installed package loop  *********

  foreach my $this_package (@installed_pkg_order)
  {
    # For each package, we decide where it will be placed and
    # update all relavant pointers.
    %{$installed_pkgs{$this_package}{target}}=();

    my $pkg_target=$installed_pkgs{$this_package}{target};  # This is the area where the output
                                                            # of the relocation
    print "\n" if ($verbose_mode);
#    my $pkg=$packages[$pkglist{$this_package}];
    # New way of declaring this so as to ween us off of the $packages list
    my $pkg=$installed_pkgs{$this_package}{ini}{ked};
    # Ok, we can now access or define keys in the current target
    # area by:     $pkg_target->{key_name}

    # The following line determines the offset based on the next_byte + the package_start_dp
    # the reason we do this is so that we can achive absolute alignment, not
    # relative alignment within the code block.
    my $basepath=$pkg->{'base_path'};
    my $code_granularity=hex($pkg->{'code_granularity'});     # Get the granularity (stored as ascii hex)
    my $variable_granularity=hex($pkg->{'variable_granularity'});
    # Make a quick reference to the memeory section
    my $pkg_memory=$installed_pkgs{$this_package}{ini}{memory};


    printf "Memory section in inifile is reporting a code size of %d\n",hex($pkg_memory->{'binary_size'})
          if ($verbose_mode && defined($pkg_memory->{'binary_size'}));
    printf "Memory section in inifile is reporting a variable size of %d\n",hex($pkg_memory->{'variable_size'})
          if ($verbose_mode && defined($pkg_memory->{'variable_size'}));

    my $looking_for_place=1;
    while ($looking_for_place)    # This variable is used by the loop to cause it to repeat
    { # ################          # if it is necessary to inc $code_target_index
      #
      # Set up the start area for the code -- adjust the cur_ptr to the code granularity
      #
      if ($target_memory[$code_target_index]{type} ne 'code')
      {
        # If the memory area is not of code type, then it is not useful so just increment
        # the index pointer
        if (++$code_target_index == $last_target)
        {
          eprintf ("Overflowed all available memory allocating code space for $this_package.\n");
          exit(100);    # The exit value of 100 comes out as 25600 to the cgi script.
        } # If we make it out of here, then we just loop and try again
        next;   # restart the while loop
      }

      my $target_mem=$target_memory[$code_target_index];  # create quick link
      my $next_byte=$target_mem->{cur_ptr};           # This is our starting point

      my $offset = ($code_granularity - ($next_byte % $code_granularity));
      $offset = 0 if ($offset==$code_granularity);    # Handle the rollover condition

      printf "Looks like we need to offset by 0x%x to meet code granularity requirement 0x%04x\n",
          $offset, $code_granularity if ($verbose_mode);

      $next_byte = $next_byte + $offset;    # adjust $next_byte with offset

      my $package_start_dp = $next_byte;    # Store the begin addr for the package
      # Ok, we have moved $next_byte forward so we can place the package
      $next_byte = $next_byte + hex($pkg_memory->{'binary_size'});

      # now we just need to test next_byte for overflow
      if ($next_byte > $target_mem->{end_ptr})
      {
        print "We've overflowed code area $code_target_index with $this_package. Trying next one. \n"
          if ($debug_relocation);
        # Test to see if we are at the end of our rope.  If so, bail
        if (++$code_target_index == $last_target)
        {
          eprintf ("Overflowed all available memory allocating code space for $this_package.\n");
          exit(100);
        } # If we make it out of here, then we just loop and try again
      }
      else
      {
        # Since it fits, we don't need to do this over again
        # Instead, we write the pointers
        $target_mem->{cur_ptr}=$next_byte; # store the changed pointer
        $target_mem->{'used'}=$target_mem->{'cur_ptr'} - $target_mem->{'start_ptr'};

        $pkg_target->{package_target_index}=$code_target_index;    # save the memory area index
        $pkg_target->{package_start_dp}=$package_start_dp;        # store the package variable

                                                        # indicating where it starts
        $looking_for_place=0;     # prevent us from repeating
        my $dp_page=hex($pkg->{'dp_page'})<<16;
        my $dp_addr=hex($pkg->{'dp_addr'});
        my $dp_xaddr=$dp_page + $dp_addr;

        $pkg_target->{'original_dp'}=$dp_xaddr;   # save this back in the hash

        if ($verbose_mode)
        {
          print "*** Placing package $this_package code as follows:\n";
          printf "This package was originally built at  dp=%06x - %06x\n",
            $dp_xaddr, $dp_xaddr + hex($pkg_memory->{'binary_size'}) ;

          printf "It's new location however is          dp=%06x - %06x\n",
            $pkg_target->{'package_start_dp'},
            ($pkg_target->{'package_start_dp'} + hex($pkg_memory->{'binary_size'}));
        }
      }
    }

    # ***********************************************************
    # ***********************************************************
    #
    # We now repeat the whole loop for variables

    $looking_for_place=1;
    while ($looking_for_place)    # This variable is used by the loop to cause it to repeat
    { # ################          # if it is necessary to inc $var_target_index
      #
      # Set up the start area for the variables -- adjust the cur_ptr to the variable granularity
      #
      if ($target_memory[$var_target_index]{type} ne 'variable')
      {
        # If the memory area is not of variable type, then it is not useful so just increment
        # the index pointer
        if (++$var_target_index == $last_target)
        {
          eprintf ("Overflowed all available memory allocating variable space for $this_package.\n");
          exit(100);
        } # If we make it out of here, then we just loop and try again
        next;   # restart the while loop
      }

      my $target_mem=$target_memory[$var_target_index];  # create quick link
      my $next_byte=$target_mem->{cur_ptr};           # This is our starting point

      my $offset = ($variable_granularity - ($next_byte % $variable_granularity));
      $offset = 0 if ($offset==$variable_granularity);    # Handle the rollover condition

      printf "Looks like we need to offset by 0x%x to meet variable granularity requirement 0x%04x\n",
          $offset, $variable_granularity if ($verbose_mode);

      $next_byte = $next_byte + $offset;    # adjust $next_byte with offset

      my $package_start_vp = $next_byte;    # Store the begin addr for the package
      # Ok, we have moved $next_byte forward so we can place the package
      $next_byte = $next_byte + hex($pkg_memory->{'variable_size'});

      # now we just need to test next_byte for overflow
      if ($next_byte > $target_mem->{end_ptr})
      {
        print "We've overflowed variable area $var_target_index with $this_package. Trying next one. \n"
          if ($debug_relocation);
        # Test to see if we are at the end of our rope.  If so, bail
        if (++$var_target_index == $last_target)
        {
          eprintf ("Overflowed all available memory allocating variable space for $this_package.\n");
          exit(100);
        } # If we make it out of here, then we just loop and try again
      }
      else
      {
        # Since it fits, we don't need to do this over again
        # Instead, we write the pointers
        $target_mem->{cur_ptr}=$next_byte; # store the changed pointer
        $target_mem->{'used'}=$target_mem->{'cur_ptr'} - $target_mem->{'start_ptr'};

       # $pkg_target->{package_target_index}=$var_target_index;  # save the memory area index
        $pkg_target->{package_start_vp}=$package_start_vp;      # store the package variable
                                                                # indicating where it starts
        $looking_for_place=0;     # allow us to exit the while loop
        my $vp_page=hex($pkg->{'vp_page'})<<16;
        my $vp_addr=hex($pkg->{'vp_addr'});
        my $vp_xaddr=$vp_page + $vp_addr;

        $pkg_target->{original_vp}=$vp_xaddr;   # This variable is so useful that we add it
                                                # back into the hash
        if ($verbose_mode)
        {
          print "Placing package $this_package variables as follows:\n";
          printf "This package was originally built at  vp=%06x - %06x\n", # vp=%02x%04x\n",
            $vp_xaddr, $vp_xaddr + hex($pkg_memory->{'variable_size'}) ;

          printf "It's new location however is          vp=%06x - %06x\n",
            $pkg_target->{'package_start_vp'},
            ($pkg_target->{'package_start_vp'} + hex($pkg_memory->{'variable_size'}));

        }
      }
    }

    # We now report the critical location information in the output files
    # here, we build an array of strings containing all location information
    # for the selected packages
    my $pkg_name_len=22;         # Width to set name/ver number to
    push @allocation_report,
      sprintf ("Package %-${pkg_name_len}s Code: %06x - %06x Vars: %06x - %06x",
        $this_package,
        $pkg_target->{'package_start_dp'},
        ($pkg_target->{'package_start_dp'} + hex($pkg_memory->{'binary_size'})),
        $pkg_target->{'package_start_vp'},
        ($pkg_target->{'package_start_vp'} + hex($pkg_memory->{'variable_size'})));

    # (re)assign the variable that will determine the dp for the library.4th
    # file or the .var_Start of the C file.  This leaves the next available
    # variable space to be set in front of the
    $next_free_vp=$pkg_target->{'package_start_vp'}+hex($pkg_memory->{'variable_size'});
    $total_vp_used+=hex($pkg_memory->{'variable_size'});  # This is just for reporting
  }
  push @allocation_report,"----End of package memory allocation listing----";
  print "Finished allocating packages\n" if ($verbose_mode);
  generate_memory_diagram();    # Generate the ascii memory usage diagram
  print "Memory usage statistics:\n";
  print "$_\n" foreach (@allocation_report);
  print "$_\n" foreach (@memory_diagram);


}   # End of allocate_packages.  This routine is where most of the magic lies.


sub build_packages()
{
# This function uses %installed_pkgs and places each package's binary code
# into a binary buffer called codeblock.  It places the packages into this
# buffer one after another and then performs the relocation according to the
# additional information specified in the ini file.

  # prior to iterating through the packages, let's format the memory.
  # this helps us avoid little 'holes' that are undefined due to granularity adjustment
  foreach my $target_index (@target_memory)
  {
    if ($target_index->{'used'})
    {
      # create an implicitly referenced anonymous list
      @{ $target_index->{'codeblock'}}=( );
      for (my $ptr=0;$ptr< $target_index->{'used'}; $ptr++)
      {
        $target_index->{'codeblock'}[$ptr]=0;
      }
    }
  }
  foreach my $this_package (@installed_pkg_order)
  { # Iterate once per package
    my $pkg=$installed_pkgs{$this_package}{ini}{ked};
    my $basepath= $pkg->{'base_path'};

    my $filename = $basepath.$pkg->{'extension_binfile'};
    my $fh;
    if (!open($fh,"<",$filename))
    {
      eprintf ("Could not open the binary file for reading, %s\n",$pkg->{'extension_binfile'});
      exit(1);
    }

    print "Linking package $this_package into target area\n";
    # ################
    #
    # Set up the start area for the code
    #

    # Read in the file to the buffer.
    my $pkg_target=$installed_pkgs{$this_package}{target};
    my $target_mem=$target_memory[$pkg_target->{package_target_index}];

    my $i=$pkg_target->{'package_start_dp'} - $target_mem->{'start_ptr'};

    while (defined(my $inchar=getc($fh)))
    {
#      printf "Storing %02x at %04x\n", ord $inchar, $i
#        if (($i<2150)&($this_package eq 'dep_test_v1.00'));
      $target_mem->{codeblock}->[$i] = ord $inchar; # ord just causes binary storage of the data
      $i++;
    }                                 # perldoc -f ord if still confused
    close($fh);
    # We've just loaded the whole binary image for this package.  Now, let's apply the
    # relocation offsets.
    my $dp_offset=$pkg_target->{'package_start_dp'} - $pkg_target->{'original_dp'};
    my $dp_page_offset=($dp_offset&0xff0000)>>16;
    my $dp_addr_offset=$dp_offset&0xffff;

    my $vp_offset=$pkg_target->{'package_start_vp'} - $pkg_target->{'original_vp'};
    my $vp_page_offset=($vp_offset&0xff0000)>>16;
    my $vp_addr_offset=$vp_offset&0xffff;
    printf "Relocating $this_package code by %06x and variables by %06x\n",$dp_offset, $vp_offset
      if ($verbose_mode);

    # We now loop through all the memory entries in the ini hash, applying the offsets
    # as we go.
    my $pkg_memory=$installed_pkgs{$this_package}{ini}{memory}; # reference to the memory section

    # These two are very important.  They are the actual shift amounts to be applied to
    # the
    my $code_shift=$pkg_target->{'package_start_dp'}-$pkg_target->{'original_dp'};
    my $var_shift=$pkg_target->{'package_start_vp'}-$pkg_target->{'original_vp'};
    my $dotcount=0;       # Used to cause auto cr for the status dots (.) printed

    foreach my $this_key (keys %{$pkg_memory})
    {
      my $package_offset=  hex($pkg_memory->{$this_key});  # The offset for the specific package

      my $target_offset= hex($pkg_memory->{$this_key})
        + $pkg_target->{'package_start_dp'} - $target_mem->{'start_ptr'};

      # one iteration per memory key.  We pattern match the key to figure out
      # what we should do.


      if ($this_key=~/^.._page/)
      { # We have a page reference.
        if ($this_key=~/^dp/)      # Within the subset of page, try dp
        { # We have a dp_page.  We've already done the general page part, not do dp specific
          # things
          printf "Ini: $this_key code target_offset: %04x pkg offset: %04x page: %02x"
            ,$target_offset ,$package_offset, $target_mem->{'codeblock'}->[$target_offset] if($debug_relocation);

          $target_mem->{'codeblock'}->[$target_offset]=$target_mem->{'codeblock'}->[$target_offset]
            + (($code_shift&0xff0000)>>16);
        }
        elsif ($this_key=~/^vp/)   # See if it is a vp_page key.
        {
          printf "Ini: $this_key  var target_offset: %04x pkg offset: %04x page: %02x"
            ,$target_offset ,$package_offset, $target_mem->{'codeblock'}->[$target_offset] if($debug_relocation);

          $target_mem->{'codeblock'}->[$target_offset]=$target_mem->{'codeblock'}->[$target_offset]
            + (($var_shift&0xff0000)>>16);
        }
        else
        { # We just make this a warning for right now
          print "Warning: Bad key, $this_key, in ini file.\n";
        }
        # PRint the changed value if debugging is on
        printf ">%02x\n",$target_mem->{'codeblock'}->[$target_offset] if($debug_relocation);
      }
      elsif ($this_key=~/^.._addr/)
      {
        my $word;

        if ($this_key=~/^dp/)      # Within the subset of addr, try dp
        { # We have a dp_addr.  We've already done the general page part, not do dp specific
          # things
          # Code related offset to be applied
          $word=   (($target_mem->{'codeblock'}->[$target_offset])<<8)
                      + $target_mem->{'codeblock'}->[$target_offset+1] + ($code_shift&0xffff);

          printf "Ini: $this_key code target_offset: %04x pkg offset: %04x addr: %04x",
            $target_offset ,$package_offset,
            (($target_mem->{'codeblock'}->[$target_offset]<<8) + $target_mem->{'codeblock'}->[$target_offset+1])
            if($debug_relocation);

          $target_mem->{'codeblock'}->[$target_offset]=$word>>8;
          $target_mem->{'codeblock'}->[$target_offset+1]=$word&0xff;
        }
        elsif ($this_key=~/^vp/)
        {       # A VAriable addr offset to be applied
          $word= ( ($target_mem->{'codeblock'}->[$target_offset]<<8)
                  + $target_mem->{'codeblock'}->[$target_offset+1] ) + ($var_shift&0xffff);

          printf "Ini: $this_key  Var target_offset: %04x pkg offset: %04x addr: %04x",
            $target_offset ,$package_offset,
            ($target_mem->{'codeblock'}->[$target_offset]<<8) + ($target_mem->{'codeblock'}->[$target_offset+1])
            if($debug_relocation);

          $target_mem->{'codeblock'}->[$target_offset]=$word>>8;
          $target_mem->{'codeblock'}->[$target_offset+1]=$word&0xff;
        }
        else
        {
          print "Warning: Bad key, $this_key, in ini file.\n";
        }
        printf ">%04x\n",
          ($target_mem->{'codeblock'}->[$target_offset]<<8) + ($target_mem->{'codeblock'}->[$target_offset+1])
          if($debug_relocation);
      }
      else
      {
       # print "Ignoring $this_key\n";
      }
      if ($verbose_mode&!$debug_relocation)
      {
        print ".";
        print "\n" if (!(++$dotcount%64));
      }
    }
    print "\n" if ($verbose_mode&!$debug_relocation);
    # END OF CODE BLOCK RELOCATION


    # BEGINNING of symbols relocation

    # *** C ADDRESSES
    my %c_pub_vars;
    if (read_ini_section( %c_pub_vars, "[c_public_variables]", $basepath.$pkg->{'package_inifile'}))
    {
      print "No public C variables section listed.  That's ok though.\n" if ($verbose_mode);
    }
    else
    { # There was a section
      # generate definitions for variables in C
      print "Generating defintions for C public variables.\n" if ($verbose_mode);
      push @c_symbols_buffer, sprintf "// Variables for Kernel Extension Package: %s$eol",$pkg->{'package_name'};
      foreach my $this_symbol (sort {hex($c_pub_vars{$a})<=>hex($c_pub_vars{$b})} keys %c_pub_vars)
      {
        my $new_addr=hex($c_pub_vars{$this_symbol}) + (($vp_page_offset<<16) + (0xffff & $vp_addr_offset));
        push @c_symbols_buffer, sprintf "#define %60s   0x%04x$eol",
          lc($this_symbol)."_XADDR", $new_addr;
      }
    }





    # Read the c function addresses
    my %c_func_hash;
    if (read_ini_section( %c_func_hash, "[c_functions]", $basepath.$pkg->{'package_inifile'}))
    {
      eprintf ("Failed reading c functions section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
      exit(1);
    }
    # Perform the relocation of the function calls
    print "Relocating c function calling addresses.\n" if ($verbose_mode);
    push @c_symbols_buffer, sprintf "// Symbols for Kernel Extension Package: %s$eol",$pkg->{'package_name'};
    my $function;
    $dotcount=0;
    foreach $function (sort {hex($c_func_hash{$a})<=>hex($c_func_hash{$b})} keys %c_func_hash)
    {
      if ($verbose_mode&!$debug_relocation)
      {
        print ".";                            # print status dots
        print "\n" if (!(++$dotcount%64));    # Auto c/r every 64 .'s
      }
      #printf "C callable function: %20s      %x -->",$function, hex($c_func_hash{$function});
      my $new_addr=hex($c_func_hash{$function}) + (($dp_page_offset<<16) + (0xffff & $dp_addr_offset));
      #printf "%06x\n",$new_addr;
      push @c_symbols_buffer, sprintf "#define %60s   0x%06x$eol",$function."_XADDR", $new_addr;
      push @c_symbols_buffer, sprintf "#define %60s   0x%04x$eol",$function."_PAGE", $new_addr>>16;
      push @c_symbols_buffer, sprintf "#define %60s   0x%04x$eol",$function."_ADDR", $new_addr&0xffff;
    }
    print "done.\n" if ($verbose_mode);

    # *** FORTH ADDRESSES
    my %forth_pub_vars;
    if (read_ini_section( %forth_pub_vars, "[forth_public_variables]", $basepath.$pkg->{'package_inifile'}))
    {
      print "No public forth variables section listed.  That's ok though.\n" if ($verbose_mode);
    }
    else
    { # There was a section
      # generate definitions for variables in forth
      print "Generating defintions for Forth public variables.\n" if ($verbose_mode);
      push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
        sprintf "\\ Variables for Kernel Extension Package: %s$eol",$pkg->{'package_name'};
      push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
        sprintf "vp x\@$eol";

      foreach my $this_symbol (sort {hex($forth_pub_vars{$a})<=>hex($forth_pub_vars{$b})} keys %forth_pub_vars)
      {
        my $new_addr=hex($forth_pub_vars{$this_symbol}) + (($vp_page_offset<<16) + (0xffff & $vp_addr_offset));
        push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
          sprintf "din %08x vp x! variable %s$eol", $new_addr, $this_symbol;
      }
      push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
        sprintf "vp x!$eol";
    }

    # Read the Forth function addresses
    my %forth_func_hash;
    if (read_ini_section( %forth_func_hash, "[forth_functions]", $basepath.$pkg->{'package_inifile'}))
    {
      eprintf ("Failed reading Forth functions section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
      exit(1);
    }

    # Perform the relocation of the function calls
    print "Relocating Forth function calling addresses.\n" if ($verbose_mode);
#    push @forth_symbols_buffer, sprintf "\\ Symbols for Kernel Extension Package: %s$eol",$pkg->{'package_name'};
    push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
      sprintf "\\ Symbols for Kernel Extension Package: %s$eol",$pkg->{'package_name'};

    # Code added 4.15.4 to fix dp allocation bug (dp must only be stacked and returned
    # for code creates, and must be counted for variable declarations).
    # So this code, and its counterpart, stack and restore the dp.
    push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
      sprintf "dp x\@$eol";

    $dotcount=0;
    foreach $function (sort {hex($forth_func_hash{$a})<=>hex($forth_func_hash{$b})} keys %forth_func_hash)
    {
      if ($verbose_mode&!$debug_relocation)
      {
        print ".";                            # print status dots
        print "\n" if (!(++$dotcount%64));    # Auto c/r every 64 .'s
      }

      #printf "Forth callable function: %20s      %x -->",$function, hex($forth_func_hash{$function});
      my $new_addr=hex($forth_func_hash{$function}) + (($dp_page_offset<<16) + (0xffff & $dp_addr_offset));
      #printf "%06x\n",$new_addr;
      push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
        sprintf "din %06x dp x! create %s$eol",$new_addr, $function;

#      push @forth_symbols_buffer, sprintf "din %06x dp x! create %s$eol",$new_addr, $function;
    }
    push @{$installed_pkgs{$this_package}{'forth_symbols_buffer'}},
      sprintf "dp x!$eol";
    print "done.\n" if ($verbose_mode);



    my @sections=list_ini_sections($basepath.$pkg->{'package_inifile'});
    # now, we parse the sections list to generate the symbols list.  The number
    # of sections is variable since unused declarations will not be given
    # a section by the keg.

    foreach my $this_section (@sections)
    { # now we test the section name to see if we're even interested in it
      if ($this_section=~/^define_/)
      { # we have a define section.  Let's list 'em
        if (read_ini_section( %{$installed_pkgs{$this_package}{'ini'}{'defines'}}, "[$this_section]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading $this_section section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
      elsif ($this_section=~/^constant_/)
      { # we have a constant section.  Let's list 'em
        if (read_ini_section( %{$installed_pkgs{$this_package}{'ini'}{'constants'}}, "[$this_section]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading $this_section section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
      elsif ($this_section=~/^l_constant_/)
      { # we have a l_constant section.  Let's list 'em
        if (read_ini_section( %{$installed_pkgs{$this_package}{'ini'}{'l_constants'}}, "[$this_section]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading $this_section section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
    }
  }
}


sub link_packages()
{
# This function iterates through the entire list of packages in the %installed_pkgs
# hash and resolves all symbols from one dependency to another.  Build_packages
# takes care of reading all the necessary information and allocating each package.
# This routine is a second pass to connect all the packages each other, where the first pass
# places the packages and internally relinks them for their actual location.


    # Do dependency resolution ++++++++++++++++++++++++++++++++++++++++++++++++
    # For each package (this loop), we must scan the kep_symbols section of the
    # hash and if it is empty, we can just do nothing, but for each
    # kep_call_addr_xxxx or kep_call_page_xxxx, we place the address or page
    # of the symbol on the right side of the equals sign into the location
    # of the current package's codeblock at the package offset indicated
    # xxxx

  print "Entering Link Package stage\n" if ($verbose_mode);
  foreach my $this_package (@installed_pkg_order)
  {
    my $pkg=$installed_pkgs{$this_package}{ini}{ked};
    my $basepath= $pkg->{'base_path'};

    my $pkg_target=$installed_pkgs{$this_package}{target};
    my $target_mem=$target_memory[$pkg_target->{package_target_index}];
    my $filename = $basepath.$pkg->{'extension_binfile'};
    my $fh;
    print "Scanning for dependency calls in package: $this_package\n" if ($verbose_mode);
    foreach my $this_dep_call (keys %{ $installed_pkgs{$this_package}{ini}{kep_symbols}})
    {
     # print "$this_dep_call\n";
      if ($this_dep_call=~/^kep_call_addr_(....)/)    # Regex extract the address of the dep call
      {
        my $dep_addr=hex($1);       # snatch the () matched portion from the key above
        # apply the proper offset for this location in the memory section
        $dep_addr +=  $installed_pkgs{$this_package}{target}{package_start_dp} -
                      $target_mem->{start_ptr};     # we need the codeblock offset so we
                                                    # subtract off the beginning of this
                                                    # package's allocated target area
        # The next line reads in the fully qualified symbol name (pkg::symbol::type)
        my $dep_symbol=$installed_pkgs{$this_package}{ini}{kep_symbols}{$this_dep_call};
        # $dep_symbol will be expressed as package_name::symbol_name.  We split on the ::
        my $dep_package;    # To contain the name of the depended on package
        my $dep_symbolname; # To contain the name of the depended on symbol
        my $type;           # to contain xcfa or xpfa indicating the type of reference
        ($dep_package, $dep_symbolname, $type)=split(/::/,$dep_symbol);  # break the fully qualified
                                                                  # symbol in two
        my $dep_symbol_addr;
        # ####################################
        # Now, we must find the xcfa or xpfa of symbolname
        # we are handling all packages as having separate name spaces
        # To find the name, we have to search 4 sections of the installed_pkgs{$dep_package} hash
        # slice.

        # 1. Test for symbol membership in the all_functions hash, and the xcfa must be requested
        if (  defined($installed_pkgs{$dep_package}{'ini'}{'all_functions'}{$dep_symbolname}) &&
              ($type eq "xcfa") )
        { # We have found a match for the requested symbol.
          $dep_symbol_addr = hex($installed_pkgs{$dep_package}{'ini'}{'all_functions'}{$dep_symbolname});
          # We have the uncorrected symbol address in $dep_symbol_addr.  Let's apply the shift
          # offset from that package's memory parameters.
          $dep_symbol_addr += $installed_pkgs{$dep_package}{'target'}{'package_start_dp'} -
                              $installed_pkgs{$dep_package}{'target'}{'original_dp'};
        }
        # 2. Test for symbol membership in the defines hash.  XCFA and XPFA requests are both handled
        elsif(defined($installed_pkgs{$dep_package}{'ini'}{'defines'}{$dep_symbolname}))
        { # It wasn't in the function list.  Lets try to look it up in the defines list

          my ($xcfa,$xpfa,$vartype)=split(/:/,$installed_pkgs{$dep_package}{'ini'}{'defines'}{$dep_symbolname});
          if ($type eq "xcfa")
          { # If we must fill in an xcfa reference...
            $dep_symbol_addr = hex($xcfa);
            # We have the uncorrected symbol address in $dep_symbol_addr.  Let's apply the shift
            # offset from that package's memory parameters.
            $dep_symbol_addr += $installed_pkgs{$dep_package}{'target'}{'package_start_dp'} -
                                $installed_pkgs{$dep_package}{'target'}{'original_dp'};
          }
          elsif ($type eq "xpfa")
          { # If we must fill in an xpfa reference...
            $dep_symbol_addr = hex($xpfa);
            # We have the uncorrected symbol address in $dep_symbol_addr.  Let's apply the shift
            # offset from that package's memory parameters.

            if ($vartype eq "code")   # If the xpfa is in the code area, then work with dp
            {
              $dep_symbol_addr += $installed_pkgs{$dep_package}{'target'}{'package_start_dp'} -
                                  $installed_pkgs{$dep_package}{'target'}{'original_dp'};
            }
            elsif ($vartype eq "var") # If the xpfa is in the variable area, the work with the vp
            {
              $dep_symbol_addr += $installed_pkgs{$dep_package}{'target'}{'package_start_vp'} -
                                  $installed_pkgs{$dep_package}{'target'}{'original_vp'};
            }
          }
        }
        # 3. Test for symbol membership in the constant's hash
        elsif (defined($installed_pkgs{$dep_package}{'ini'}{'constants'}{$dep_symbolname}))
        {
          my ($xcfa,$value)=split(/:/,$installed_pkgs{$dep_package}{'ini'}{'constants'}{$dep_symbolname});
          $dep_symbol_addr = hex($xcfa);
          $dep_symbol_addr += 1 if ($type eq "xpfa");   # If the xpfa is called for on a constant, which
                                                        # is very unusual, then we simply shift the xcfa f
                                                        # forward some.
          # We have the uncorrected symbol address in $dep_symbol_addr.  Let's apply the shift
          # offset from that package's memory parameters.
          $dep_symbol_addr += $installed_pkgs{$dep_package}{'target'}{'package_start_dp'} -
                                  $installed_pkgs{$dep_package}{'target'}{'original_dp'};
        }
        # 4. Test for symbol membership in the l_constant's hash
        elsif (defined($installed_pkgs{$dep_package}{'ini'}{'l_constants'}{$dep_symbolname}))
        {
          my ($xcfa,$value)=split(/:/,$installed_pkgs{$dep_package}{'ini'}{'l_constants'}{$dep_symbolname});
          $dep_symbol_addr = hex($xcfa);
          $dep_symbol_addr += 1 if ($type eq "xpfa");   # If the xpfa is called for on a constant, which
                                                        # is very unusual, then we simply shift the xcfa f
                                                        # forward some.
          # We have the uncorrected symbol address in $dep_symbol_addr.  Let's apply the shift
          # offset from that package's memory parameters.
          $dep_symbol_addr += $installed_pkgs{$dep_package}{'target'}{'package_start_dp'} -
                                  $installed_pkgs{$dep_package}{'target'}{'original_dp'};
        }
        else  # By now, if one of these statements hasn't evaluated true, then there's a problem
        {
          eprintf ("Got a symbol we couldn't resolve: %s \n", $dep_symbol);
          exit (1);
        }

        printf "Code addr offset: %06x Target area: %02x  %s::%s::%s Symbol Addr: %06x\n",
          $dep_addr, $pkg_target->{package_target_index},$dep_package, $dep_symbolname,
          $type,$dep_symbol_addr   if ($debug_dependency);
        my $newdata=$dep_symbol_addr&0xffff;  # strip off the page information
        my $olddata=($target_mem->{'codeblock'}->[$dep_addr]<<8) + $target_mem->{'codeblock'}->[$dep_addr+1];
        printf "Old data= %04x   New data= %04x\n",$olddata,$newdata  if ($debug_dependency);

        # now lets actually do it!
        $target_mem->{'codeblock'}->[$dep_addr]=($dep_symbol_addr&0xffff)>>8;      # store high byte
        $target_mem->{'codeblock'}->[$dep_addr+1]=($dep_symbol_addr&0xffff)&0xff;  # store lowbyte

      }

      # Now, we do the dependency pages ++++++++++++++++++++
      # if we find a kep_call_page_*, we do basically the same thing as above, except it
      # happens only on a per package basis, not per symbol.
      elsif ($this_dep_call=~/^kep_call_page_(....)/)    # Regex extract the page of the dep call
      {
        my $dep_addr=hex($1);       # snatch the () matched portion from the key above
        # apply the proper offset for this location in the memory section
        $dep_addr +=  $installed_pkgs{$this_package}{'target'}{'package_start_dp'} -
                      $target_mem->{'start_ptr'};     # we need the codeblock offset so we
                                                    # subtract off the beginning of this
                                                    # package's allocated target area

        my $dep_package=$installed_pkgs{$this_package}{'ini'}{'kep_symbols'}{$this_dep_call};

        if (!defined($installed_pkgs{$dep_package}))
        {
          eprintf ("Got a dependency package page we couldn't resolve: %s \n", $dep_package);
          exit (1);
        }
        else
        { # We have record of it..
          my $dep_package_page =
            ($installed_pkgs{$dep_package}{'target'}{'package_start_dp'} & 0xff0000)>>16;

          printf "Code page offset: %06x Target area: %02x  %s Symbol Addr: %06x\n",
            $dep_addr, $pkg_target->{'package_target_index'},$dep_package, $dep_package_page
            if ($debug_dependency);
          my $olddata=  $target_mem->{'codeblock'}->[$dep_addr];
          my $newdata=$dep_package_page&0xff;  # strip off the page information
          printf "Old data= %04x   New data= %04x\n",$olddata,$newdata  if ($debug_dependency);
          $target_mem->{'codeblock'}->[$dep_addr]=$newdata;
        }
      }
    }

  }
}



sub write_symlist()
{
# This function uses %installed_pkgs to produce a simple line delimited
# list of all the would be create names, that is, all forth callable
# functions.

  my $this_pkg;
  foreach $this_pkg (@installed_pkg_order)
  { # Iterate once per package.
    print "**** Looking at package: $this_pkg\n" if ($verbose_mode);
    my $pkg=$packages[$pkglist{$this_pkg}];
                                        # pkglist is a hash that translates
                                        # the name of the package to its index
                                        # in the packages array.  The packages
                                        # array contains info for all the
                                        # packages
    # Read the Forth function addresses
    my %all_func_hash;  # Functions
    my %defines;        # Words that have pfas
    my %l_constants;    # Long constants
    my %constants;      # Constants

    my $basepath= $pkg->{'base_path'};

    my @sections=list_ini_sections($basepath.$pkg->{'package_inifile'});
    # now, we parse the sections list to generate the symbols list.  The number
    # of sections is variable since unused declarations will not be given
    # a section by the keg.

    foreach my $this_section (@sections)
    { # now we test the section name to see if we're even interested in it
      if ($this_section=~/^define_/)
      { # we have a define section.  Let's list 'em
        if (read_ini_section( %defines, "[$this_section]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading $this_section section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
      elsif ($this_section=~/^constant_/)
      { # we have a constant section.  Let's list 'em
        if (read_ini_section( %constants, "[$this_section]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading $this_section section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
      elsif ($this_section=~/^l_constant_/)
      { # we have a l_constant section.  Let's list 'em
        if (read_ini_section( %l_constants, "[$this_section]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading $this_section section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
      elsif ($this_section eq "all_functions")
      {
        if (read_ini_section( %all_func_hash, "[all_functions]", $basepath.$pkg->{'package_inifile'}))
        {
          eprintf ("Failed reading all functions section from inifile:%s\n",$basepath.$pkg->{'package_inifile'});
          exit(1);
        }
      }
    }

    print "Beginning of package: $pkg->{'package_name'}_v$pkg->{'package_version'}\n";
    foreach my $function (keys (%all_func_hash))
    {
      print "Code Symbol: $function\n";
    }
    foreach my $define (keys (%defines))
    {
      print "Defining Symbol: $define\n";
    }
    foreach my $constant (keys (%constants))
    {
      $constants{$constant} =~ /^.*:(.*)/;
      print "Constant Symbol: $constant=$1\n";
    }
    foreach my $constant (keys (%l_constants))
    {
      $l_constants{$constant} =~ /^.*:(.*)/;
      print "L_Constant Symbol: $constant=$1\n";
    }

    print "**** done with that package.\n" if ($verbose_mode);

  }
}



# ######################   Output file handlers below   ###########
# ####
# ####

sub append_file(\@$)
# Usage:    append_file($array_ref, $filename)
# The purpose of this function is to attempt to open the specified filename
# and if it is successfully opened, it places the file and returns with
# a zero exit status.  If the file cannot be opened, then it will return
# nonzero. This is handy for preambles and postambles.
{
  my $buffer=$_[0];
  my $filename=$_[1];
  my $fh;

  if (open($fh,"<$filename"))
  {
    print "Reading $filename.." if ($verbose_mode);
    while (<$fh>)
    {
      s/\r|\n//g;                  # Remove any eol char
      push @$buffer,$_.$eol;       # Add it to the buffer
    }
    print "..done\n" if ($verbose_mode);
    close($fh);
    return(0);      # Since we were successful, we can exit 0
  }
  else
  {
    print "Didn't find $filename.  Probably not important.\n" if ($verbose_mode);
    return(1);      # As promised, return nonzero
  }
}


sub write_install_file()
{
  # This routine writes out the S record portion of the 4th install file.

  print "Writing the install file\n" if ($verbose_mode);

  # First we place the system preamble file.  The one thing I can think of
  # right now that belongs in this file is the download.map instruction
  if (append_file(@install_file_buffer,$pkg_source_path.$install_pre_filename))
  {
    print "No system preamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # Now insert the allocation report
  push @install_file_buffer, "\\".($_?" $_":"").$eol foreach (@allocation_report);
  # Place the memory sumary in there as well
  push @install_file_buffer, "\\".($_?" $_":"").$eol foreach (@memory_diagram);
  # Rotate through the packages checking to an install preamble file in
  # each package path.
  foreach (@installed_pkg_order)
  {
    if (
      !append_file (@install_file_buffer,
        $packages[$pkglist{$_}]->{'base_path'}.$install_pre_filename)
       )
    {
      printf "Found install preamble file for package: %s\n",
        $packages[$pkglist{$_}]->{'package_name'} if ($verbose_mode);
    }
  }

  # Now it's time to generate the S records for the codeblocks in the @target_memory array.
  my $targetcount=-1;     # Yup, we have to inc it first thing through the loop
                          # That's cuz of the 'next' statements that cause the loop to restart
  foreach my $this_target (@target_memory)
  {
    $targetcount++;
    next if ($this_target->{'type'} ne 'code');   # skip all non code target areas
    next if ($this_target->{'used'}==0);           # skip nonused target areas
    print "Writing codeblock for target area $targetcount\n" if ($verbose_mode);

    push @install_file_buffer,(sprintf "hex din %06x receive.hex$eol",$s_rec_dest_xaddr);

    # Now for the middle stuff.   This is the actual memory image after
    # relocation and cancatenations of all the requested packages.
    # create S record block now that all the preambles are done.

    # Usage:  generate_s_block ( @binary_buffer, $binary_buffer_offset, $length.
    #         $s_addr_base )

    my @new_s_recs = generate_s_block( @{$this_target->{'codeblock'}}, 0x000000,
      $this_target->{'used'}, 0x000000);    # Generate the S records into the buffer

    foreach (@new_s_recs)                   # current position
    {                                       # Process the end of line
      s/\r|\n//g;                             # Remove any eol char
      push @install_file_buffer,$_.$eol;      # Store line to buffer
    }
    # This adds what comes immediately after the S records
    push @install_file_buffer,
      (sprintf
        "hex din %06x din %06x %04x $copy_directive$eol$eol",
        $s_rec_dest_xaddr, $this_target->{'start_ptr'}, $this_target->{'used'});
  }

  # Rotate through the packages checking for an install postamble file in
  # each package path.
  foreach (@installed_pkg_order)
  {
    if (
      !append_file (@install_file_buffer,
        $packages[$pkglist{$_}]->{'base_path'}.$install_post_filename)
       )
    {
      printf "Found install postamble file for package: %s\n",
        $packages[$pkglist{$_}]->{'package_name'} if ($verbose_mode);
    }
  }

  # now do the system postamble
  if (append_file(@install_file_buffer,$pkg_source_path.$install_post_filename))
  {
    print "No system postamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # The buffer's all ready - lets write it to the file
  open_outfile('install_file');    # Open the filehandle for the sfile
  foreach (@install_file_buffer)
  {
    print {$filestreams{'install_file'}} $_;     # Save it to a file
  }
}


sub write_forth_file()
{
  # This routine writes the Forth Library file.

  print "Writing the Forth library file\n" if ($verbose_mode);

  # First we place the system preamble file.  The one thing I can think of
  # right now that belongs in this file is the download.map instruction
  if (append_file(@forth_file_buffer,$pkg_source_path.$forth_pre_filename))
  {
    print "No system forth preamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # Now insert the allocation report
  push @forth_file_buffer, "\\".($_?" $_":"").$eol foreach (@allocation_report);

  # Lets also throw in the memory diagram for Forth users
  push @forth_file_buffer, "\\".($_?" $_":"").$eol foreach (@memory_diagram);

  # We need to set the vp to start at the end of the variables
  # used by the kernel extensions
  # Ok,  Now we're finished with the create statements.
  # We need to do the same thing for Forth.
  push @forth_file_buffer,
      sprintf "$eol\\ The Kernel Extensions used 0x%04x actual bytes of variable space$eol",
      $total_vp_used;
  push @forth_file_buffer,
      sprintf "base @ hex din %06x vp x! base !$eol$eol", $next_free_vp;


  # Rotate through the packages checking for an raw forth file in
  # each package path.  We must now go through each package and insert the
  # ini file specified forth
  foreach my $this_pkg (@installed_pkg_order)
  {
    push @forth_file_buffer, sprintf "base @ hex$eol"; # Save our current address
    foreach (@{$installed_pkgs{$this_pkg}{'forth_symbols_buffer'}})
    {
      push @forth_file_buffer, $_; # Line endings already supplied
    }
    push @forth_file_buffer, sprintf "base !$eol";   # put the candle back

    my $fullfilepath=$packages[$pkglist{$this_pkg}]->{'base_path'}.
      $packages[$pkglist{$this_pkg}]->{'raw_forthfile'};

    push @forth_file_buffer,
      sprintf $eol."\\ Beginning raw forthfile for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    if ( !append_file(@forth_file_buffer, $fullfilepath))
    { # success (function returned 0) condition
      printf "Inserting raw Forth file for package: %s\n",
        $packages[$pkglist{$this_pkg}]->{'package_name'} if ($verbose_mode);
    }
    push @forth_file_buffer,
      sprintf $eol."\\ End of raw forthfile for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
 # this should not cause and error.  taken out by jlw on 5.29.2
 #    else
 #    { #  File not found (function returned nonzero) condition
 #     eprintf ("Could not open file the raw forthfile, %s, specified by the package: %s\n",
 #       $packages[$pkglist{$this_pkg}]->{'raw_forthfile'},
 #       $packages[$pkglist{$this_pkg}]->{'package_name'});
 #     eprint "Raw forthfiles are a required part of Kernel Extensions. Terminating\n";
 #     exit(1);
 #   }
  }

  # Store next available byte in last code region used,
  # in case application wants to start defining words there.
  push @forth_file_buffer,
    sprintf "base @ hex din %06x xconstant end.of.mosaic.kx.code base !$eol",$target_memory[$code_target_index]{cur_ptr};

  # now do the system postamble
  if (append_file(@forth_file_buffer,$pkg_source_path.$forth_post_filename))
  {
    print "No system forth postamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # The buffer's all ready - lets write it to the file
  open_outfile('forth_file');     # Open the filehandle for the file
  foreach (@forth_file_buffer)
  {
    print {$filestreams{'forth_file'}} $_;     # Save it to a file
  }
}


sub write_c_file()
{
  # This routine writes the C source file.

  print "Writing C source file\n" if ($verbose_mode);


  # This routine writes the C Source File

  # First we place the system preamble file.
  if (append_file(@c_source_file_buffer,$pkg_source_path.$sys_c_src_pre_filename))
  {
    print "No system C Source preamble file present.  Skipping.\n" if ($verbose_mode);
  }
  # Now insert the allocation report
  push @c_source_file_buffer, "//".($_?" $_":"").$eol foreach (@allocation_report);

  # Rotate through the packages checking for an raw C file in
  # each package path.  We must now go through each package and insert the
  # ini file specified C
  foreach my $this_pkg (@installed_pkg_order)
  {
    my $fullfilepath=$packages[$pkglist{$this_pkg}]->{'base_path'}.
      $packages[$pkglist{$this_pkg}]->{'c_sourcefile'};

    push @c_source_file_buffer,
      sprintf $eol."// Beginning C Source file for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    if ( !append_file(@c_source_file_buffer, $fullfilepath))
    { # success (function returned 0) condition
      printf "Inserting C Source file for package: %s\n",
        $packages[$pkglist{$this_pkg}]->{'package_name'} if ($verbose_mode);
    push @c_source_file_buffer,
      sprintf $eol."// End of C Source file for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    }
    else
    { #  File not found (function returned nonzero) condition
      # We are making this not be an error anymore 7.22.3
    #  eprintf ("Could not open file the C Source file, %s, specified by the package: %s\n",
    #    $packages[$pkglist{$this_pkg}]->{'c_sourcefile'},
    #    $packages[$pkglist{$this_pkg}]->{'package_name'});
    #  eprint "C Source files are a required part of Kernel Extensions. Terminating\n";
    #  exit(1);
      push @c_source_file_buffer,
        "// No C callable functions in package $packages[$pkglist{$this_pkg}]->{'package_name'}$eol";
    }
  }

  # First we place the system postamble file.
  if (append_file(@c_source_file_buffer,$pkg_source_path.$sys_c_src_post_filename))
  {
    print "No system C Source postamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # The buffer's all ready - lets write it to the file
  open_outfile('c_source_file');
  foreach (@c_source_file_buffer)
  {
    print {$filestreams{'c_source_file'}} $_;     # Save it to a file
  }
}


sub write_h_file()
{
  # This routine writes the C header file.

  print "Writing C header file\n" if ($verbose_mode);

  # This routine writes the C Header File

  # First we place the system preamble file.
  if (append_file(@c_header_file_buffer,$pkg_source_path.$sys_c_hdr_pre_filename))
  {
    print "No system C Header preamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # Now insert the allocation report
  push @c_header_file_buffer, "//".($_?" $_":"").$eol foreach (@allocation_report);

  # Now place the .VarStart pointer setup
  push @c_header_file_buffer,
    sprintf "#pragma option nowarn=dup$eol";
  push @c_header_file_buffer,
    sprintf "#asm$eol";
  push @c_header_file_buffer,
    sprintf ".VarStart:  equ  \$%04x$eol",$next_free_vp;
  push @c_header_file_buffer,
    sprintf "#endasm$eol";
  push @c_header_file_buffer,
    sprintf "#pragma option warn=dup$eol";


  foreach (@c_symbols_buffer)
  {
    #print {$filestreams{'forth_file'}} $_;   # write the line out
    push @c_header_file_buffer, $_; # Line endings already supplied
  }

  # Rotate through the packages checking for an raw H file in
  # each package path.  We must now go through each package and insert the
  # ini file specified H
  foreach my $this_pkg (@installed_pkg_order)
  {
    my $fullfilepath=$packages[$pkglist{$this_pkg}]->{'base_path'}.
      $packages[$pkglist{$this_pkg}]->{'c_headerfile'};

    push @c_header_file_buffer,
      sprintf $eol."// Beginning C Header file for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    if ( !append_file(@c_header_file_buffer, $fullfilepath))
    { # success (function returned 0) condition
      printf "Inserting C Header file for package: %s\n",
        $packages[$pkglist{$this_pkg}]->{'package_name'} if ($verbose_mode);
    push @c_header_file_buffer,
      sprintf $eol."// End of C Header file for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    }
    else
    { #  File not found (function returned nonzero) condition
      # eprintf ("Could not open the C Header file, %s, specified by the package: %s\n",
      #  $packages[$pkglist{$this_pkg}]->{'c_headerfile'},
      #  $packages[$pkglist{$this_pkg}]->{'package_name'});
      # eprint "C Header files are a required part of Kernel Extensions. Terminating\n";
      # exit(1);
      push @c_header_file_buffer,
        "// No C callable functions in package $packages[$pkglist{$this_pkg}]->{'package_name'}$eol";
    }
  }

  # First we place the system postamble file.
  if (append_file(@c_header_file_buffer,$pkg_source_path.$sys_c_hdr_post_filename))
  {
    print "No system C Header postamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # The buffer's all ready - lets write it to the file
  open_outfile('c_header_file');
  foreach (@c_header_file_buffer)
  {
    print {$filestreams{'c_header_file'}} $_;     # Save it to a file
  }

}



sub write_readme_file()
{
  # This routine writes the readme file.

  print "Writing readme file\n" if ($verbose_mode);

  # Now insert the allocation report
  push @readme_file_buffer, $_.$eol foreach (@allocation_report);
  push @readme_file_buffer, $_.$eol foreach (@memory_diagram);

  # This routine writes the readme File

  # First we place the system preamble file.
  if (append_file(@readme_file_buffer,$pkg_source_path.$sys_readme_pre_filename))
  {
    print "No system readme preamble file present.  Skipping.\n" if ($verbose_mode);
  }
  # Rotate through the packages checking for a readme file in
  # each package path.  We must now go through each package and insert the
  # ini file specified file
  foreach my $this_pkg (@installed_pkg_order)
  {
    my $fullfilepath=$packages[$pkglist{$this_pkg}]->{'base_path'}.
      $packages[$pkglist{$this_pkg}]->{'readmefile'};

    if (! -r $fullfilepath)
    { # Skip packages with no readme files
      printf "No package readme file, %s, present for package: %s.  Skipping.\n",
        $packages[$pkglist{$this_pkg}]->{'readmefile'},
        $packages[$pkglist{$this_pkg}]->{'package_name'} if ($verbose_mode);
      next;
    }
    push @readme_file_buffer,
      sprintf "$eol$eol** Beginning readme file for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    if ( !append_file(@readme_file_buffer, $fullfilepath))
    { # success (function returned 0) condition
      printf "Inserting readme file for package: %s\n",
        $packages[$pkglist{$this_pkg}]->{'package_name'} if ($verbose_mode);
    push @readme_file_buffer,
      sprintf "** End of readme file for package: %s$eol",$packages[$pkglist{$this_pkg}]->{'package_name'};
    }
    else
    { #  File not found (function returned nonzero) condition
      eprintf ("Could not open the readme file, %s, specified by the package: %s\n",
        $filenames{'readme_file'},
        $packages[$pkglist{$this_pkg}]->{'package_name'});
      eprint "This error should have never happened since readme files are optional. Terminating\n";
      exit(1);
    }
  }

  # First we place the system postamble file.
  if (append_file(@readme_file_buffer,$pkg_source_path.$sys_readme_post_filename))
  {
    print "No system readme postamble file present.  Skipping.\n" if ($verbose_mode);
  }

  # The buffer's all ready - lets write it to the file
  open_outfile('readme_file');
  foreach (@readme_file_buffer)
  {
    print {$filestreams{'readme_file'}} $_;     # Save it to a file
  }
}


sub write_files()
{
  # This function writes thes buffers out to the different files.  Each file
  # its own function to handle opening, writing, and closing it.

  print "Writing output files.\n" if ($verbose_mode);
  write_install_file();
  write_forth_file();
  write_c_file();
  write_h_file();
  write_readme_file();
  print "Finished writing output files.\n" if ($verbose_mode);
}



# ######################   Stream handlers below   ###########################
# ####
# ####

sub get_dirlist()
{
  # This function scans the package source path and populates
  # the array, ini_listing, with a list of all ini filepaths
  # relative to the package source path.  This function
  # does not test the ini files for any additional validity,
  # but ensures that they are normal files.  It also does not
  # descend directory paths.  If it is decided that this is needed,
  # this function needs to be modified.  The ONLY level that is considered
  # is the level 1 below the package source path.  Since this is a web
  # application, recursion is generally considered to be reckless when
  # it comes to security.  Note also that we verify that the files are
  # regular files so that symlinks are not followed or considered.

  my $dirhandle;
  if (!opendir($dirhandle,$pkg_source_path))
  {
    eprint "Can't open directory, $pkg_source_path\n";
    exit (1);
  }
  # Ok, we got it open
  my $file;
  my $pkgfile;
  my $pkgdir;
  #my $count=0;
  while (defined($file=readdir($dirhandle)))
  {   # Itterate through each file in the source path
    next if (($file eq ".") || ($file eq ".."));
    my $current_file=$pkg_source_path.$file."/";  # make this for convienience
    # Test to see if we have a directory
    if (-d $current_file)
    { # It is a directory, so lets search it for .ini's
      print "Found directory: $file\n" if ($debug_list);
      if (opendir(my $pkgdir, $current_file))
      {
        while(defined($pkgfile=readdir($pkgdir)))
        {   # Iterate through for each file in this directory
          if (($pkgfile =~ /\.ini/) && (-f $current_file.$pkgfile))
          {   # If it is a regular file and ends with .ini
            push @ini_listing,$file."/".$pkgfile;
          }
        }
      }
    }
  }
}


sub open_outfile($)
{
# Inputs: name of hash member for filehandle to be opened

# Checks to see if the specified member of the %filestreams hash has been
# opened.
  my $in_name=$_[0];            # This is the name of the filestreams hash
                                # element that contains the file pointer
                                # reference for the filehandle we are opening
                                # This name is also that of the element of the
                                # option hash that contains the actual
                                # filename.
  my $myref;

  # Adding a test and create for kemout if it doesn't exist
  if (! -e $outpath)  # check for/create the local basepath
  {
    print "Output directory $outpath missing.  I'm creating it for you\n"
      if ($verbose_mode);
    mkdir ($outpath);
  }


  # my FH;
  local *FH;
  if (!$filestreams{$in_name})
  {     # If the filestream is not open..
    $outpath =~ m/^([a-zA-z0-9-._+=\/]+)$/;
    my $outfilepath = $1;

    $filenames{$in_name} =~ m/^([a-zA-z0-9-._+=]+)$/;
    # print "\n\n$1\n\n";
    $outfilepath .= "/".$1;


    if (!open(FH, ">","$outfilepath"))
    {  # We failed to open the file
      error "Couldn't $outfilepath for writing\n";
      exit(1);    # We couldn't open it
    }
    $filestreams{$in_name}=*FH;
    # $filestreams{$in_name}=*OUTSTREAM;
  }
  # now, the filestream has either just been successfully opened or it was
  # already open.
  return;
}


sub close_outfiles()
{
  my $this_stream;
  foreach $this_stream (keys %filestreams)
  {
    if ($filestreams{$this_stream})
    {
      printf "Closing output filename: %s\n",$filenames{$this_stream} if ($verbose_mode);
      close($filestreams{$this_stream});
      $filestreams{$this_stream}=0;
    }
  }
}


sub eprint(@)
  # This is a lower level error print function that gives me the option
  # of later printing to more than one stream.
{
  print $errorstream @_;
}


sub eprintf(@)
  # This is a lower level error print function that gives me the option
  # of later printing to more than one stream.
{
  printf $errorstream @_;
}


sub error(@)
  # Use this just like print, except that this will print it to the
  # appropriate error stream(s).
{
  eprint @_;    # Print the error text
}

sub crunch_cmd_options()
{
  # parse for valid command line arguments

  # scalar @argv contains the number of arguments.  Parse based on that.
  if (scalar(@ARGV)==0)
  { # Test the number of arguments
    eprint("You don't have any arguments.  Try kem --help\n");
    eprint("Entering interactive mode in 2 seconds\n");
    sleep(2);
    $command="interactive";
  } else {
    $command=$ARGV[0];
  }

  if (lc($command) =~ /(--help)|(-h)|(help)/)
  {
    print $usage_info_text;
    print $help_text;
    exit(0);    # Exit with no error after showing help
  }

  # Now that we have the command, lets remove it from the list so that
  # the getopt calls that follow will not be confused.
  for (my $count=0;$count<scalar(@ARGV)-1;$count++)
  {
    $ARGV[$count]=$ARGV[$count+1];
  }   # Scoot all the arguments up by one slot

  delete($ARGV[scalar(@ARGV)-1]);     # Delete the last element

  # Ok, now we're ready to begin normal parsing of the line

  Getopt::Long::Configure("pass_through");  # we don't implement all options in
                                            # the POSIX style
  GetOptions( "help"=>\$cmd_options{h},
              "verbose"=>\$cmd_options{v},
              "quiet"=>\$cmd_options{q},
              "version"=>\$cmd_options{V});

  getopts("vqVhao:s:u:",\%cmd_options);  # This is gonna set up the cmd_options
                                  # in a hash.
                                  # After we're all done here, @ARGV will contain
                                  # only the operands that aren't options

  if ($cmd_options{v}) { $verbose_mode=1 }  # Test for verbose flag
  else { $verbose_mode=0 }

  print "Verbose mode on\n" if ($verbose_mode);


  if ($cmd_options{a}) { $show_all=1 }  # Test for show all flag
  else { $show_all=0 }

  # ##########
  # ##########
  # ##########    Handle the -o option
  # ##########

  if ($cmd_options{o} && $cmd_options{o} =~ m|^(/tmp/[^.~]+)$| )  # Check for the -o option
  {
    $outpath=$1;  # Set the output path
  }

  # ##########
  # ##########
  # ##########    Handle the -q option (quiet)
  # ##########        It should be noted that this routine should run early
  # ##########        since it prints the program ID banner.  We want the
  # ##########        banner to be shown before anything else.

  if (!$cmd_options{q})
  {
    print $prog_id_text;
    $quiet=0;
  }
  else
  {
    $quiet=1;   # Shhhh
  }

  # lowlevel debugging stuff...
  if ($lowlevel_cmdline)
  {
    printf "Options are V: %s v: %s q: %s h: %s s: %s \n",
      ($cmd_options{V} ? "yes " : "no "),
      ($cmd_options{v} ? "yes " : "no "),
      ($cmd_options{q} ? "yes " : "no "),
      ($cmd_options{h} ? "yes " : "no "),
      ($cmd_options{a} ? "yes " : "no "),
      ($cmd_options{s} ? "yes: $pkg_source_path " : "no "),
      ($cmd_options{u} ? "yes: $cmd_options{u} " : "no "),
      ($cmd_options{o} ? "yes: $outpath " : "no ");

   # print ("Argvs: 0:$ARGV[0] 1:$ARGV[1] \n");
  }


  # ##########
  # ##########
  # ##########    Handle the -h option (help)
  # ##########

  # Test for help flag and possibly exit
  if ($cmd_options{h})
  {
    print $usage_info_text;
    print $help_text;
    exit(0);    # Exit with no error after showing help
  }


  # ##########
  # ##########
  # ##########    Handle the -V option (show version and exit)
  # ##########      Print the ID and exit.  Put this before -q to ensure
  # ##########      that the banner is not redundantly printed twice.

  # Test for Version flag and possibly exit
  if ($cmd_options{V})
  {
    print $prog_id_text;
    exit(0);    # Exit with no error
  }


  # If we have not yet exited due to -h/--help, or -V/--version then prepare
  # the variables.

  # ##########
  # ##########
  # ##########    Handle the -u option
  # ##########

  my $user_config_file;

  if ($cmd_options{u})  # Check for the -u option
  {
    $user_config_file=$cmd_options{u};  # Set the user config file accordingly
  }
  else
  {
    die "Config file must be specified with -u option.";
  }

  # untaint the command line specifiable string
  if ($user_config_file =~ /^([\/\-\@\w.]+)$/)
  {
    $user_config_file = $1;
  }
  else
  {     # the string didn't pass muster.  Bail now!
    die "Fishy string specified for the user config file: $user_config_file";
  }

  # read the user configuration file.  Note that this code is only reachable
  # when the -S option is NOT used.
  if (-e "$user_config_file")
  {
    print "Reading user configuration: $user_config_file." if ($verbose_mode);
    do "$user_config_file";
    print ".done\n" if ($verbose_mode);
  }
  else
  {
    die "Your specified user config file, $user_config_file, cannot be opened!\n";
  }

  # ##########
  # ##########
  # ##########    Handle the -s option
  # ##########      cmd line pkg path trumps config file path

  if ($cmd_options{s})  # Check for the -s option
  {
    $pkg_source_path=$cmd_options{s};  # Set the package path
  }
  # untaint the command line specifiable string
  if ($pkg_source_path =~ /^([\/\-\@\w.]+)$/)
  {
    $pkg_source_path = $1;
  }
  else
  {     # the string didn't pass muster.  Bail now!
    die "Fishy string specified for the package source path";
  }
  $pkg_source_path .= "/" if (!($pkg_source_path =~/\/$/));
}

