#!/usr/bin/perl -T -w
# MRP 2002 - 11/04
#   PriceSheet.cgi -- Converts a CSV Price Sheet into a HTML Document. The CSV
# records are formated using a table. Used by the corporate website to provide
# price info for the products. This script optional can take a query string that
# only contains a Part#. That Part# is then used to display the corresponding
# record at the top of the pricesheet as a highlighted table row.
#   Comma Separated Values is a simple data storage protocol originally designed
# by MicroSoft. The records are delimited by lines, and the fields are delimited
# by commas. If a field contains commas or begins and/or ends with spaces, the
# field must be surround by double quotes. If a quoted field contains quotes,
# those quotes must be escaped with another quote.
#   Example CSV Record: [" a, ", b b  , "c"""]
#            Parsed As: [ a, ][b b][c"]
# More info about CSV available at http://www.xbeat.net/vbspeed/c_ParseCSV.php
#   This script expects a CSV Record containing: [Part#, Description, Price, 100+]
#-------------------------------------------------------------------------------
use strict;         # Enable Perl's strict rules; this enhances security and
                    # reduces bugs
use warnings;
use CGI qw(escapeHTML);
delete @ENV{qw(IFS PATH CDPATH ENV BASH_ENV)};  # Clean up security parils
                                                # These variables could be used
                                                # to alter how the script runs,
                                                # so we clear them.
# Global variables declared here (required by strict mode)
our $csv_FileName = "./Price.csv";
our $PartID = '';
if (defined $ENV{'QUERY_STRING'})
{
  $PartID = $ENV{'QUERY_STRING'};
  $PartID =~ tr/+/ /;
  $PartID =~ s/%([2-9A-F][0-9A-F])/pack("C", hex($1))/egi;
}

our $QuoteParity = 0;   # States Quote Parity (Even, Odd)
our $WhiteSpace = "";
our @Fields = ();
our $Field = "";
our $FieldEnd = "";     # States Field Type (unknown, Quoted, not Quoted)

our %Products;          # Declare the products hash


# Due to strict mode, all subroutines must be declared
sub Parse_CSV_Price_Sheet($);    # Loads the Price Sheet into the Products Hash
sub Split_CSV_Record($);         # Parses a line from the CSV file into an Array
sub Add_Field();                 # Helper function for Split_CSV_Record($)
sub Write_HTML_Preamble();
sub Write_HTML_Price_Table();
sub Write_HTML_Postamble();


# Main runtime starts here
   Parse_CSV_Price_Sheet($csv_FileName);
   Write_HTML_Preamble();
   Write_HTML_Price_Table();
   Write_HTML_Postamble();

# Main runtime ends here ----  Subroutines follow
#===============================================================================

#===============================================================================
# Open CSV format Price Sheet and parse each record into a
# Hash that is Keyed by Part#.Desc and Part#.Price
#-------------------------------------------------------------------------------
sub Parse_CSV_Price_Sheet($) {
  my ($Line, $Key, @Field);

  open (hCSV,"<","$_[0]") or die "Cannot open the data file"; # Open CSV File
  while ($Line = <hCSV>) {                      # Read in next Line
   $Line =~ s/[\r\n]+$//;                       # Strip Line Endings
   (@Field) = Split_CSV_Record($Line);          # Parse CSV Record
   if ($Field[0] eq "MIPartID") {next}          # Drop Field Names
   $Key = $Field[0];
   $Products{$Key}{Desc}  = $Field[1];
   $Products{$Key}{Price} = $Field[2];
   $Products{$Key}{Ext}   = $Field[3];
   }
  close (hCSV);
}
#===============================================================================

#===============================================================================
# Parse a CSV Record and Return with a list of the Fields
# Expects a Scalar containing the CSV Record
# Returns an Array containing the CSV Fields
#-------------------------------------------------------------------------------
sub Split_CSV_Record($) {
   my $Record = $_[0];
   my $Char;            # Declare locals...

   $QuoteParity = 0;    # States Quote Parity  true=inside quotes/false=outside quotes
   $WhiteSpace = "";    # Empty whitespace
   @Fields = ();
   $FieldEnd = "";

   foreach $Char (split //, $Record) {
      unless ($FieldEnd) {                         # Discover Field Type
         next if ($Char =~ /\s/);                  # Remove leading WhiteSpace
         if    ($Char eq ',') {Add_Field()}        # Field is Empty
         elsif ($Char eq '"') {$FieldEnd = '"'}    # Field is Quoted
         else {                                    # Field is not Quoted
            $FieldEnd = ',';
            $Field .= $Char;                       # Add Char to Field
            }
         }
      elsif ($FieldEnd eq '"') {                   # Parse for Quoted Field
         if ($QuoteParity) {                       # Previous Char was '"'
            if ($Char eq ',') {Add_Field()}        # Quoted Field has Ended
            elsif ($Char eq '"') {                 # Escaped Quotes Found
               $Field .= $Char;                    # Add Quote to Field
               $QuoteParity=0;                     # Reset Quote Parity
               }
            else {                                 # End of Quoted Section
               $FieldEnd = ',';                    # Change Field Type
               redo;                               # ReParse as non Quoted Field
               }
            }
         elsif ($Char eq '"') {$QuoteParity = 1}   # Quote Encountered
         else {$Field .= $Char}                    # Add Char to Field
         }
      elsif ($Char eq "," ) {Add_Field()}          # Non Quoted Field has Ended
      elsif ($Char =~ /\s/) {$WhiteSpace .= $Char} # Separate WhiteSpace
      else {                                       # Non WhiteSpace found,
         $Field = $Field.$WhiteSpace.$Char;        #    Recombine with Field
         $WhiteSpace = "";                         # Reset WhiteSpace Buffer
         }
      }
   Add_Field();                                    # Add last Field to List
   return @Fields;                                 # Return with List of Fields
}
#-----------------------------
sub Add_Field() {
   push @Fields, $Field;                           # Add Field to List
   $FieldEnd = $Field = "";                        # Reset State Machine
   $WhiteSpace = "";                               # Variables
   $QuoteParity = 0;                               #
}
#===============================================================================

#===============================================================================
# Write, to Standard Out, the beginning of the HTML document
#-------------------------------------------------------------------------------
sub Write_HTML_Preamble() {
print <<"End_Of_HTML";
Content-Type: text/html

<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">
<html>
<head>
   <title>Mosaic Industries - Price Sheet</title>
   <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
   <meta name="keywords" content="mosaic industries, embedded control, embedded controller, embedded controllers, embedded systems, programmable controller, programmable controllers, single board computer, single board computers, operator interface, operator interfaces, scientific instruments, portable instruments, industrial control, embedded applications, graphical user interface, GUI, data acquisition, I/O modules, signal conditioning, AC and DC relays, touchscreen-controlled, real-time control, analog and digital I/O, SCADA">
   <meta name="description" content="Low cost single board computers, microcontroller boards, industrial controllers, HMI/MMI operator interfaces, & touch screen graphical user interfaces (GUIs), modular I/O for embedded systems, computer I-O, industrial control, data acquisition, ethernet & communications in instrumentation and automation.">
   <link rel="icon" href="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAACXBIWXMAAA7EAAAOxAGVKw4bAAAARElEQVR42mNgoA743wDBMDbxgAm3gfQF6F4g3jtMxBk+sF7A5R0In4l0i2jjLWK9gMkm4AXGhoFOSITVkBAL6N6hkvcAKa86aKbjO6UAAAAASUVORK5CYII=">
   <link rel="stylesheet" type="text/css" href="../Style.css">
   <link rel="stylesheet" type="text/css" href="/Banner.css">
   <style type="text/css"><!--
TABLE.PriceSheet {
   font:             8pt Arial;
   border:           1pt Solid Black;
   background:       White;
   margin:           10px;
   width:            600px;
   }
TH.PriceSheet {
   background:       Black;
   color:            Yellow;
   }
TD {
   border-bottom:    1pt Solid Black;
   margin:           0;
   padding:          2pt;
   }
TD.Price {
   text-align:       Right;
   }
TR.First {
   color:            Red;
   font-weight:      600;
   border-bottom:    2pt Solid Red;
   }
--></style>
</head>
<!-- This File is Dynamically Generated by /PriceSheet.cgi -->
 <body>
 <!--******Logo & Tabs Section******************-->
 <table class="Banner" cellspacing="0" cellpadding="0"><tr>
    <td id="Logo1"><img src="/Logo/webbanner_logo1.jpg" alt="low cost single board computers, embedded controllers, and operator interfaces for scientific instruments & industrial control"></td>
    </tr><tr>
      <td id="Tabs1"><img src="/MenuBars/TabProd.gif" usemap="#tabs_all" alt="">
<map name="tabs_all">
       <area shape="rect" coords="002,11,097,31" href="/"             alt="manufacturer of I/O-rich SBCs, operator interfaces, handheld instruments, and development tools for embedded control">
       <area shape="rect" coords="100,11,195,31" href="/Products/"    alt="low cost single board computers, embedded controllers, and operator interfaces for scientific instruments & industrial control">
       <area shape="rect" coords="200,11,295,31" href="/Resources/"   alt="development tools for embedded control">
       <area shape="rect" coords="299,11,402,31" href="/Order/"       alt="order our low cost I/O-rich embedded control products">
       <area shape="rect" coords="405,11,502,31" href="/About/"       alt="embedded controller manufacturer profile">
</map>
      </td>
    </tr><tr>
      <td id="SubMenu"><img src="/MenuBars/prod_submenu_default.gif" usemap="#prod_submenu" alt="">
<map name="prod_submenu">
   <area shape="rect" coords="001,1,094,21" href="/Products/SBC/"          alt="single board computers & embedded controllers">
   <area shape="rect" coords="099,1,174,21" href="/Products/Packages/"     alt="development tools & starter kits for your embedded design">
   <area shape="rect" coords="177,1,254,21" href="/Products/GUI/"          alt="operator interfaces with touchscreens and graphical user interface">
   <area shape="rect" coords="259,1,352,21" href="/Products/WildCards/"    alt="plug-in expansion modules for digital & analog I/O">
   <area shape="rect" coords="357,1,422,21" href="/Products/Software/"     alt="C language & Forth language integrated development tools, IDE">
   <area shape="rect" coords="426,1,509,21" href="/Products/Acc/"          alt="single board and embedded computer accessories">
   <area shape="rect" coords="512,1,591,21" href="/Products/Enclosures/"   alt="embedded controller enclosures, bezels, environmental gaskets">
</map>
      </td>
    </tr></table>

<table><!--******************main content table begins***********************-->
<!-- **************left hand menu************-->
<tr><td width="120px" valign="top" align="center" bgcolor="#E4E7EF">

<script type="text/javascript" src="/Menu/menu950_var.js"></script>
<script type="text/javascript" src="/Menu/menu950_com.js"></script>

<noscript>
   <p>
   <a class="NoScr" href="/">Home</a><br>
   <a class="NoScr" href="/Products/GUI/">Embedded GUIs</a><br>
   <a class="NoScr" href="/Products/SBC/">Single Board Computers</a><br>
   <a class="NoScr" href="/Products/WildCards/">"Expansion I/O modules</a><br>
   <a class="NoScr" href="/Products/Packages/">Design Kits</a><br>
   <a class="NoScr" href="/Resources/Appnotes.html">Application Notes</a><br>
   <a class="NoScr" href="/Products/Software/index.html">Software</a><br>
   <a class="NoScr" href="/Products/Acc/index.html">Accessories</a><br>
   <a class="NoScr" href="/Resources/Faqs.html">FAQ</a><br>
   <a class="NoScr" href="/embedded-systems/">Documentation</a><br>
   <a class="NoScr" href="/Resources/Links.html">Links</a><br>
   <a class="NoScr" href="/About/index.html">Contact Us</a><br>
   Copyright (c) 2003 Mosaic Industries, Inc.
   </p>
</noscript>
</td>

<td valign="top" >


<!--******[ Content Starts ]************************-->
<div class="Desc">

<h3>Products and Prices</h3>
<h4>Notice</h4>
<p>The pricing and availability of the following products may change at any time without notice. Furthermore, this pricesheet may not reflect those changes.</p>
<p><a href="http://www.mosaic-industries.com/embedded-systems/dokuwiki/help-for-readers/contact-us">Contact Mosaic Industries</a> to confirm pricing and discounts.</p>
<br><p><b>Have questions about Mosaic products? Need help with your project? Want to chat?</b> Please check us out at <a href="/embedded-systems/">Mosaic Documentation Web</a>...</p>

</div>
<table class="PriceSheet">
<tr style="background: Black">
   <th class="PriceSheet">Part #</th>
   <th class="PriceSheet">Description</th>
   <th class="PriceSheet">Price:</th>
   <th class="PriceSheet">100 Price:</th>
</tr>
End_Of_HTML
}
#===============================================================================

#===============================================================================
# Write, to Standard Out, a HTML Table Row for each element
# of the Products Hash.
#-------------------------------------------------------------------------------
sub Write_HTML_Price_Table() {
  my $Key;
  my @prodlist = sort keys %Products;
  if (defined $Products{$PartID}) { unshift(@prodlist, $PartID); }
  foreach $Key (@prodlist) {
    my $p = $Products{$Key};
    my $Desc  = defined $p->{'Desc'}  && $p->{'Desc'}  ? escapeHTML($p->{'Desc'})  : '&mdash;';
    my $Price = defined $p->{'Price'} && $p->{'Price'} ? escapeHTML($p->{'Price'}) : '&mdash;';
    my $Ext   = defined $p->{'Ext'}   && $p->{'Ext'}   ? escapeHTML($p->{'Ext'})   : '&mdash;';
    my $class_attr = $Key eq $PartID ? ' class="First"' : '';
    print '
<tr', $class_attr, '>
   <td class="Part">',  $Key,   '</td>
   <td class="Desc">',  $Desc,  '</td>
   <td class="Price">', $Price, '</td>
   <td class="Price">', $Ext,   '</td>
</tr>
';
   }
}

#===============================================================================
# Write, to Standard Out, the closure of the HTML document
#-------------------------------------------------------------------------------
sub Write_HTML_Postamble() { print '</table></td></tr></table></body></html>'; }
#===============================================================================
